/*-
 * Copyright (c) 2008 Stephen L. Moshier <steve@moshier.net>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*							cpowl
 *
 *	Complex power function
 *
 *
 *
 * SYNOPSIS:
 *
 * long double complex cpowl();
 * long double complex a, z, w;
 *
 * w = cpowl (a, z);
 *
 *
 *
 * DESCRIPTION:
 *
 * Raises complex A to the complex Zth power.
 * Definition is per AMS55 # 4.2.8,
 * analytically equivalent to cpow(a,z) = cexp(z clog(a)).
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -10,+10     30000       9.4e-15     1.5e-15
 *
 */

#include <sys/cdefs.h>
#include <complex.h>
#include <math.h>
#include "math_private.h"

long double complex
cpowl(long double complex a, long double complex z)
{
	long double complex w;
	long double x, y, r, theta, absa, arga;

	x = creall(z);
	y = cimagl(z);
	absa = cabsl(a);
	if (absa == 0.0L) {
		return (CMPLXL(0.0L, 0.0L));
	}
	arga = cargl(a);
	r = powl(absa, x);
	theta = x * arga;
	if (y != 0.0L) {
		r = r * expl(-y * arga);
		theta = theta + y * logl(absa);
	}
	w = CMPLXL(r * cosl(theta), r * sinl(theta));
	return (w);
}
