/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 2005  Joseph Koshy
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef _DEV_HWPMC_PMC_EVENTS_H_
#define	_DEV_HWPMC_PMC_EVENTS_H_

/*
 * Note: Documentation on adding events can be found online at:
 *
 * https://wiki.freebsd.org/PmcTools/PmcHardwareHowTo
 *
 * Please refer to this resource before you attempt to modify this file or the
 * hwpmc driver/subsystem.
 */

/* * PMC event codes.
 *
 * __PMC_EV(CLASS, SYMBOLIC-NAME)
 *
 */
/* timestamp counters. */
#define        __PMC_EV_TSC()                                                  \
	__PMC_EV(TSC, TSC)

#define        PMC_EV_TSC_FIRST        PMC_EV_TSC_TSC
#define        PMC_EV_TSC_LAST         PMC_EV_TSC_TSC

#define	__PMC_EV_ALIAS_TSC()			\
__PMC_EV_ALIAS("cycles",	TSC_TSC)

/*
 * Software events are dynamically defined.
 */

#define        PMC_EV_DYN_COUNT        0x1000

#define        PMC_EV_SOFT_FIRST       0x20000
#define        PMC_EV_SOFT_LAST        (PMC_EV_SOFT_FIRST + PMC_EV_DYN_COUNT - 1)

/* AMD K8 PMCs */

#define	__PMC_EV_K8()							\
__PMC_EV(K8, FP_DISPATCHED_FPU_OPS)					\
__PMC_EV(K8, FP_CYCLES_WITH_NO_FPU_OPS_RETIRED)				\
__PMC_EV(K8, FP_DISPATCHED_FPU_FAST_FLAG_OPS)				\
__PMC_EV(K8, LS_SEGMENT_REGISTER_LOAD)					\
__PMC_EV(K8, LS_MICROARCHITECTURAL_RESYNC_BY_SELF_MODIFYING_CODE)	\
__PMC_EV(K8, LS_MICROARCHITECTURAL_RESYNC_BY_SNOOP)			\
__PMC_EV(K8, LS_BUFFER2_FULL)						\
__PMC_EV(K8, LS_LOCKED_OPERATION)					\
__PMC_EV(K8, LS_MICROARCHITECTURAL_LATE_CANCEL)				\
__PMC_EV(K8, LS_RETIRED_CFLUSH_INSTRUCTIONS)				\
__PMC_EV(K8, LS_RETIRED_CPUID_INSTRUCTIONS)				\
__PMC_EV(K8, DC_ACCESS)							\
__PMC_EV(K8, DC_MISS)							\
__PMC_EV(K8, DC_REFILL_FROM_L2)						\
__PMC_EV(K8, DC_REFILL_FROM_SYSTEM)					\
__PMC_EV(K8, DC_COPYBACK)						\
__PMC_EV(K8, DC_L1_DTLB_MISS_AND_L2_DTLB_HIT)				\
__PMC_EV(K8, DC_L1_DTLB_MISS_AND_L2_DTLB_MISS)				\
__PMC_EV(K8, DC_MISALIGNED_DATA_REFERENCE)				\
__PMC_EV(K8, DC_MICROARCHITECTURAL_LATE_CANCEL)				\
__PMC_EV(K8, DC_MICROARCHITECTURAL_EARLY_CANCEL)			\
__PMC_EV(K8, DC_ONE_BIT_ECC_ERROR)					\
__PMC_EV(K8, DC_DISPATCHED_PREFETCH_INSTRUCTIONS)			\
__PMC_EV(K8, DC_DCACHE_ACCESSES_BY_LOCKS)				\
__PMC_EV(K8, BU_CPU_CLK_UNHALTED)					\
__PMC_EV(K8, BU_INTERNAL_L2_REQUEST)					\
__PMC_EV(K8, BU_FILL_REQUEST_L2_MISS)					\
__PMC_EV(K8, BU_FILL_INTO_L2)						\
__PMC_EV(K8, IC_FETCH)							\
__PMC_EV(K8, IC_MISS)							\
__PMC_EV(K8, IC_REFILL_FROM_L2)						\
__PMC_EV(K8, IC_REFILL_FROM_SYSTEM)					\
__PMC_EV(K8, IC_L1_ITLB_MISS_AND_L2_ITLB_HIT)				\
__PMC_EV(K8, IC_L1_ITLB_MISS_AND_L2_ITLB_MISS)				\
__PMC_EV(K8, IC_MICROARCHITECTURAL_RESYNC_BY_SNOOP)			\
__PMC_EV(K8, IC_INSTRUCTION_FETCH_STALL)				\
__PMC_EV(K8, IC_RETURN_STACK_HIT)					\
__PMC_EV(K8, IC_RETURN_STACK_OVERFLOW)					\
__PMC_EV(K8, FR_RETIRED_X86_INSTRUCTIONS)				\
__PMC_EV(K8, FR_RETIRED_UOPS)						\
__PMC_EV(K8, FR_RETIRED_BRANCHES)					\
__PMC_EV(K8, FR_RETIRED_BRANCHES_MISPREDICTED)				\
__PMC_EV(K8, FR_RETIRED_TAKEN_BRANCHES)					\
__PMC_EV(K8, FR_RETIRED_TAKEN_BRANCHES_MISPREDICTED)			\
__PMC_EV(K8, FR_RETIRED_FAR_CONTROL_TRANSFERS)				\
__PMC_EV(K8, FR_RETIRED_RESYNCS)					\
__PMC_EV(K8, FR_RETIRED_NEAR_RETURNS)					\
__PMC_EV(K8, FR_RETIRED_NEAR_RETURNS_MISPREDICTED)			\
__PMC_EV(K8, FR_RETIRED_TAKEN_BRANCHES_MISPREDICTED_BY_ADDR_MISCOMPARE)	\
__PMC_EV(K8, FR_RETIRED_FPU_INSTRUCTIONS)				\
__PMC_EV(K8, FR_RETIRED_FASTPATH_DOUBLE_OP_INSTRUCTIONS)		\
__PMC_EV(K8, FR_INTERRUPTS_MASKED_CYCLES)				\
__PMC_EV(K8, FR_INTERRUPTS_MASKED_WHILE_PENDING_CYCLES)			\
__PMC_EV(K8, FR_TAKEN_HARDWARE_INTERRUPTS)				\
__PMC_EV(K8, FR_DECODER_EMPTY)						\
__PMC_EV(K8, FR_DISPATCH_STALLS)					\
__PMC_EV(K8, FR_DISPATCH_STALL_FROM_BRANCH_ABORT_TO_RETIRE)		\
__PMC_EV(K8, FR_DISPATCH_STALL_FOR_SERIALIZATION)			\
__PMC_EV(K8, FR_DISPATCH_STALL_FOR_SEGMENT_LOAD)			\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_REORDER_BUFFER_IS_FULL)		\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_RESERVATION_STATIONS_ARE_FULL)	\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_FPU_IS_FULL)			\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_LS_IS_FULL)				\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_WAITING_FOR_ALL_TO_BE_QUIET)	\
__PMC_EV(K8, FR_DISPATCH_STALL_WHEN_FAR_XFER_OR_RESYNC_BRANCH_PENDING)	\
__PMC_EV(K8, FR_FPU_EXCEPTIONS)						\
__PMC_EV(K8, FR_NUMBER_OF_BREAKPOINTS_FOR_DR0)				\
__PMC_EV(K8, FR_NUMBER_OF_BREAKPOINTS_FOR_DR1)				\
__PMC_EV(K8, FR_NUMBER_OF_BREAKPOINTS_FOR_DR2)				\
__PMC_EV(K8, FR_NUMBER_OF_BREAKPOINTS_FOR_DR3)				\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_PAGE_ACCESS_EVENT)			\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_PAGE_TABLE_OVERFLOW)			\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_DRAM_COMMAND_SLOTS_MISSED)		\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_TURNAROUND)				\
__PMC_EV(K8, NB_MEMORY_CONTROLLER_BYPASS_SATURATION)			\
__PMC_EV(K8, NB_SIZED_COMMANDS)						\
__PMC_EV(K8, NB_PROBE_RESULT)						\
__PMC_EV(K8, NB_HT_BUS0_BANDWIDTH)					\
__PMC_EV(K8, NB_HT_BUS1_BANDWIDTH)					\
__PMC_EV(K8, NB_HT_BUS2_BANDWIDTH)

#define	PMC_EV_K8_FIRST		PMC_EV_K8_FP_DISPATCHED_FPU_OPS
#define	PMC_EV_K8_LAST		PMC_EV_K8_NB_HT_BUS2_BANDWIDTH

/*
 * Events supported by Intel architectural fixed function counters,
 * from the "Intel 64 and IA-32 Architectures Software Developer's
 * Manual Volume 3B: System Programming Guide, Part 2", July 2008.
 */
#define	__PMC_EV_IAF()							\
__PMC_EV(IAF, INSTR_RETIRED_ANY)					\
__PMC_EV(IAF, CPU_CLK_UNHALTED_CORE)					\
__PMC_EV(IAF, CPU_CLK_UNHALTED_REF)

#define	PMC_EV_IAF_FIRST	PMC_EV_IAF_INSTR_RETIRED_ANY
#define	PMC_EV_IAF_LAST		PMC_EV_IAF_CPU_CLK_UNHALTED_REF

#define	__PMC_EV_ALIAS_IAF()						\
__PMC_EV_ALIAS("instruction-retired",	IAF_INSTR_RETIRED_ANY)		\
__PMC_EV_ALIAS("unhalted-core-cycles",	IAF_CPU_CLK_UNHALTED_CORE)	\
__PMC_EV_ALIAS("unhalted-reference-cycles", IAF_CPU_CLK_UNHALTED_REF)

/*
 * ARMv7 Events
 */

#define	__PMC_EV_ARMV7()			\
	__PMC_EV(ARMV7, EVENT_00H)		\
	__PMC_EV(ARMV7, EVENT_01H)		\
	__PMC_EV(ARMV7, EVENT_02H)		\
	__PMC_EV(ARMV7, EVENT_03H)		\
	__PMC_EV(ARMV7, EVENT_04H)		\
	__PMC_EV(ARMV7, EVENT_05H)		\
	__PMC_EV(ARMV7, EVENT_06H)		\
	__PMC_EV(ARMV7, EVENT_07H)		\
	__PMC_EV(ARMV7, EVENT_08H)		\
	__PMC_EV(ARMV7, EVENT_09H)		\
	__PMC_EV(ARMV7, EVENT_0AH)		\
	__PMC_EV(ARMV7, EVENT_0BH)		\
	__PMC_EV(ARMV7, EVENT_0CH)		\
	__PMC_EV(ARMV7, EVENT_0DH)		\
	__PMC_EV(ARMV7, EVENT_0EH)		\
	__PMC_EV(ARMV7, EVENT_0FH)		\
	__PMC_EV(ARMV7, EVENT_10H)		\
	__PMC_EV(ARMV7, EVENT_11H)		\
	__PMC_EV(ARMV7, EVENT_12H)		\
	__PMC_EV(ARMV7, EVENT_13H)		\
	__PMC_EV(ARMV7, EVENT_14H)		\
	__PMC_EV(ARMV7, EVENT_15H)		\
	__PMC_EV(ARMV7, EVENT_16H)		\
	__PMC_EV(ARMV7, EVENT_17H)		\
	__PMC_EV(ARMV7, EVENT_18H)		\
	__PMC_EV(ARMV7, EVENT_19H)		\
	__PMC_EV(ARMV7, EVENT_1AH)		\
	__PMC_EV(ARMV7, EVENT_1BH)		\
	__PMC_EV(ARMV7, EVENT_1CH)		\
	__PMC_EV(ARMV7, EVENT_1DH)		\
	__PMC_EV(ARMV7, EVENT_1EH)		\
	__PMC_EV(ARMV7, EVENT_1FH)		\
	__PMC_EV(ARMV7, EVENT_20H)		\
	__PMC_EV(ARMV7, EVENT_21H)		\
	__PMC_EV(ARMV7, EVENT_22H)		\
	__PMC_EV(ARMV7, EVENT_23H)		\
	__PMC_EV(ARMV7, EVENT_24H)		\
	__PMC_EV(ARMV7, EVENT_25H)		\
	__PMC_EV(ARMV7, EVENT_26H)		\
	__PMC_EV(ARMV7, EVENT_27H)		\
	__PMC_EV(ARMV7, EVENT_28H)		\
	__PMC_EV(ARMV7, EVENT_29H)		\
	__PMC_EV(ARMV7, EVENT_2AH)		\
	__PMC_EV(ARMV7, EVENT_2BH)		\
	__PMC_EV(ARMV7, EVENT_2CH)		\
	__PMC_EV(ARMV7, EVENT_2DH)		\
	__PMC_EV(ARMV7, EVENT_2EH)		\
	__PMC_EV(ARMV7, EVENT_2FH)		\
	__PMC_EV(ARMV7, EVENT_30H)		\
	__PMC_EV(ARMV7, EVENT_31H)		\
	__PMC_EV(ARMV7, EVENT_32H)		\
	__PMC_EV(ARMV7, EVENT_33H)		\
	__PMC_EV(ARMV7, EVENT_34H)		\
	__PMC_EV(ARMV7, EVENT_35H)		\
	__PMC_EV(ARMV7, EVENT_36H)		\
	__PMC_EV(ARMV7, EVENT_37H)		\
	__PMC_EV(ARMV7, EVENT_38H)		\
	__PMC_EV(ARMV7, EVENT_39H)		\
	__PMC_EV(ARMV7, EVENT_3AH)		\
	__PMC_EV(ARMV7, EVENT_3BH)		\
	__PMC_EV(ARMV7, EVENT_3CH)		\
	__PMC_EV(ARMV7, EVENT_3DH)		\
	__PMC_EV(ARMV7, EVENT_3EH)		\
	__PMC_EV(ARMV7, EVENT_3FH)		\
	__PMC_EV(ARMV7, EVENT_40H)		\
	__PMC_EV(ARMV7, EVENT_41H)		\
	__PMC_EV(ARMV7, EVENT_42H)		\
	__PMC_EV(ARMV7, EVENT_43H)		\
	__PMC_EV(ARMV7, EVENT_44H)		\
	__PMC_EV(ARMV7, EVENT_45H)		\
	__PMC_EV(ARMV7, EVENT_46H)		\
	__PMC_EV(ARMV7, EVENT_47H)		\
	__PMC_EV(ARMV7, EVENT_48H)		\
	__PMC_EV(ARMV7, EVENT_49H)		\
	__PMC_EV(ARMV7, EVENT_4AH)		\
	__PMC_EV(ARMV7, EVENT_4BH)		\
	__PMC_EV(ARMV7, EVENT_4CH)		\
	__PMC_EV(ARMV7, EVENT_4DH)		\
	__PMC_EV(ARMV7, EVENT_4EH)		\
	__PMC_EV(ARMV7, EVENT_4FH)		\
	__PMC_EV(ARMV7, EVENT_50H)		\
	__PMC_EV(ARMV7, EVENT_51H)		\
	__PMC_EV(ARMV7, EVENT_52H)		\
	__PMC_EV(ARMV7, EVENT_53H)		\
	__PMC_EV(ARMV7, EVENT_54H)		\
	__PMC_EV(ARMV7, EVENT_55H)		\
	__PMC_EV(ARMV7, EVENT_56H)		\
	__PMC_EV(ARMV7, EVENT_57H)		\
	__PMC_EV(ARMV7, EVENT_58H)		\
	__PMC_EV(ARMV7, EVENT_59H)		\
	__PMC_EV(ARMV7, EVENT_5AH)		\
	__PMC_EV(ARMV7, EVENT_5BH)		\
	__PMC_EV(ARMV7, EVENT_5CH)		\
	__PMC_EV(ARMV7, EVENT_5DH)		\
	__PMC_EV(ARMV7, EVENT_5EH)		\
	__PMC_EV(ARMV7, EVENT_5FH)		\
	__PMC_EV(ARMV7, EVENT_60H)		\
	__PMC_EV(ARMV7, EVENT_61H)		\
	__PMC_EV(ARMV7, EVENT_62H)		\
	__PMC_EV(ARMV7, EVENT_63H)		\
	__PMC_EV(ARMV7, EVENT_64H)		\
	__PMC_EV(ARMV7, EVENT_65H)		\
	__PMC_EV(ARMV7, EVENT_66H)		\
	__PMC_EV(ARMV7, EVENT_67H)		\
	__PMC_EV(ARMV7, EVENT_68H)		\
	__PMC_EV(ARMV7, EVENT_69H)		\
	__PMC_EV(ARMV7, EVENT_6AH)		\
	__PMC_EV(ARMV7, EVENT_6BH)		\
	__PMC_EV(ARMV7, EVENT_6CH)		\
	__PMC_EV(ARMV7, EVENT_6DH)		\
	__PMC_EV(ARMV7, EVENT_6EH)		\
	__PMC_EV(ARMV7, EVENT_6FH)		\
	__PMC_EV(ARMV7, EVENT_70H)		\
	__PMC_EV(ARMV7, EVENT_71H)		\
	__PMC_EV(ARMV7, EVENT_72H)		\
	__PMC_EV(ARMV7, EVENT_73H)		\
	__PMC_EV(ARMV7, EVENT_74H)		\
	__PMC_EV(ARMV7, EVENT_75H)		\
	__PMC_EV(ARMV7, EVENT_76H)		\
	__PMC_EV(ARMV7, EVENT_77H)		\
	__PMC_EV(ARMV7, EVENT_78H)		\
	__PMC_EV(ARMV7, EVENT_79H)		\
	__PMC_EV(ARMV7, EVENT_7AH)		\
	__PMC_EV(ARMV7, EVENT_7BH)		\
	__PMC_EV(ARMV7, EVENT_7CH)		\
	__PMC_EV(ARMV7, EVENT_7DH)		\
	__PMC_EV(ARMV7, EVENT_7EH)		\
	__PMC_EV(ARMV7, EVENT_7FH)		\
	__PMC_EV(ARMV7, EVENT_80H)		\
	__PMC_EV(ARMV7, EVENT_81H)		\
	__PMC_EV(ARMV7, EVENT_82H)		\
	__PMC_EV(ARMV7, EVENT_83H)		\
	__PMC_EV(ARMV7, EVENT_84H)		\
	__PMC_EV(ARMV7, EVENT_85H)		\
	__PMC_EV(ARMV7, EVENT_86H)		\
	__PMC_EV(ARMV7, EVENT_87H)		\
	__PMC_EV(ARMV7, EVENT_88H)		\
	__PMC_EV(ARMV7, EVENT_89H)		\
	__PMC_EV(ARMV7, EVENT_8AH)		\
	__PMC_EV(ARMV7, EVENT_8BH)		\
	__PMC_EV(ARMV7, EVENT_8CH)		\
	__PMC_EV(ARMV7, EVENT_8DH)		\
	__PMC_EV(ARMV7, EVENT_8EH)		\
	__PMC_EV(ARMV7, EVENT_8FH)		\
	__PMC_EV(ARMV7, EVENT_90H)		\
	__PMC_EV(ARMV7, EVENT_91H)		\
	__PMC_EV(ARMV7, EVENT_92H)		\
	__PMC_EV(ARMV7, EVENT_93H)		\
	__PMC_EV(ARMV7, EVENT_94H)		\
	__PMC_EV(ARMV7, EVENT_95H)		\
	__PMC_EV(ARMV7, EVENT_96H)		\
	__PMC_EV(ARMV7, EVENT_97H)		\
	__PMC_EV(ARMV7, EVENT_98H)		\
	__PMC_EV(ARMV7, EVENT_99H)		\
	__PMC_EV(ARMV7, EVENT_9AH)		\
	__PMC_EV(ARMV7, EVENT_9BH)		\
	__PMC_EV(ARMV7, EVENT_9CH)		\
	__PMC_EV(ARMV7, EVENT_9DH)		\
	__PMC_EV(ARMV7, EVENT_9EH)		\
	__PMC_EV(ARMV7, EVENT_9FH)		\
	__PMC_EV(ARMV7, EVENT_A0H)		\
	__PMC_EV(ARMV7, EVENT_A1H)		\
	__PMC_EV(ARMV7, EVENT_A2H)		\
	__PMC_EV(ARMV7, EVENT_A3H)		\
	__PMC_EV(ARMV7, EVENT_A4H)		\
	__PMC_EV(ARMV7, EVENT_A5H)		\
	__PMC_EV(ARMV7, EVENT_A6H)		\
	__PMC_EV(ARMV7, EVENT_A7H)		\
	__PMC_EV(ARMV7, EVENT_A8H)		\
	__PMC_EV(ARMV7, EVENT_A9H)		\
	__PMC_EV(ARMV7, EVENT_AAH)		\
	__PMC_EV(ARMV7, EVENT_ABH)		\
	__PMC_EV(ARMV7, EVENT_ACH)		\
	__PMC_EV(ARMV7, EVENT_ADH)		\
	__PMC_EV(ARMV7, EVENT_AEH)		\
	__PMC_EV(ARMV7, EVENT_AFH)		\
	__PMC_EV(ARMV7, EVENT_B0H)		\
	__PMC_EV(ARMV7, EVENT_B1H)		\
	__PMC_EV(ARMV7, EVENT_B2H)		\
	__PMC_EV(ARMV7, EVENT_B3H)		\
	__PMC_EV(ARMV7, EVENT_B4H)		\
	__PMC_EV(ARMV7, EVENT_B5H)		\
	__PMC_EV(ARMV7, EVENT_B6H)		\
	__PMC_EV(ARMV7, EVENT_B7H)		\
	__PMC_EV(ARMV7, EVENT_B8H)		\
	__PMC_EV(ARMV7, EVENT_B9H)		\
	__PMC_EV(ARMV7, EVENT_BAH)		\
	__PMC_EV(ARMV7, EVENT_BBH)		\
	__PMC_EV(ARMV7, EVENT_BCH)		\
	__PMC_EV(ARMV7, EVENT_BDH)		\
	__PMC_EV(ARMV7, EVENT_BEH)		\
	__PMC_EV(ARMV7, EVENT_BFH)		\
	__PMC_EV(ARMV7, EVENT_C0H)		\
	__PMC_EV(ARMV7, EVENT_C1H)		\
	__PMC_EV(ARMV7, EVENT_C2H)		\
	__PMC_EV(ARMV7, EVENT_C3H)		\
	__PMC_EV(ARMV7, EVENT_C4H)		\
	__PMC_EV(ARMV7, EVENT_C5H)		\
	__PMC_EV(ARMV7, EVENT_C6H)		\
	__PMC_EV(ARMV7, EVENT_C7H)		\
	__PMC_EV(ARMV7, EVENT_C8H)		\
	__PMC_EV(ARMV7, EVENT_C9H)		\
	__PMC_EV(ARMV7, EVENT_CAH)		\
	__PMC_EV(ARMV7, EVENT_CBH)		\
	__PMC_EV(ARMV7, EVENT_CCH)		\
	__PMC_EV(ARMV7, EVENT_CDH)		\
	__PMC_EV(ARMV7, EVENT_CEH)		\
	__PMC_EV(ARMV7, EVENT_CFH)		\
	__PMC_EV(ARMV7, EVENT_D0H)		\
	__PMC_EV(ARMV7, EVENT_D1H)		\
	__PMC_EV(ARMV7, EVENT_D2H)		\
	__PMC_EV(ARMV7, EVENT_D3H)		\
	__PMC_EV(ARMV7, EVENT_D4H)		\
	__PMC_EV(ARMV7, EVENT_D5H)		\
	__PMC_EV(ARMV7, EVENT_D6H)		\
	__PMC_EV(ARMV7, EVENT_D7H)		\
	__PMC_EV(ARMV7, EVENT_D8H)		\
	__PMC_EV(ARMV7, EVENT_D9H)		\
	__PMC_EV(ARMV7, EVENT_DAH)		\
	__PMC_EV(ARMV7, EVENT_DBH)		\
	__PMC_EV(ARMV7, EVENT_DCH)		\
	__PMC_EV(ARMV7, EVENT_DDH)		\
	__PMC_EV(ARMV7, EVENT_DEH)		\
	__PMC_EV(ARMV7, EVENT_DFH)		\
	__PMC_EV(ARMV7, EVENT_E0H)		\
	__PMC_EV(ARMV7, EVENT_E1H)		\
	__PMC_EV(ARMV7, EVENT_E2H)		\
	__PMC_EV(ARMV7, EVENT_E3H)		\
	__PMC_EV(ARMV7, EVENT_E4H)		\
	__PMC_EV(ARMV7, EVENT_E5H)		\
	__PMC_EV(ARMV7, EVENT_E6H)		\
	__PMC_EV(ARMV7, EVENT_E7H)		\
	__PMC_EV(ARMV7, EVENT_E8H)		\
	__PMC_EV(ARMV7, EVENT_E9H)		\
	__PMC_EV(ARMV7, EVENT_EAH)		\
	__PMC_EV(ARMV7, EVENT_EBH)		\
	__PMC_EV(ARMV7, EVENT_ECH)		\
	__PMC_EV(ARMV7, EVENT_EDH)		\
	__PMC_EV(ARMV7, EVENT_EEH)		\
	__PMC_EV(ARMV7, EVENT_EFH)		\
	__PMC_EV(ARMV7, EVENT_F0H)		\
	__PMC_EV(ARMV7, EVENT_F1H)		\
	__PMC_EV(ARMV7, EVENT_F2H)		\
	__PMC_EV(ARMV7, EVENT_F3H)		\
	__PMC_EV(ARMV7, EVENT_F4H)		\
	__PMC_EV(ARMV7, EVENT_F5H)		\
	__PMC_EV(ARMV7, EVENT_F6H)		\
	__PMC_EV(ARMV7, EVENT_F7H)		\
	__PMC_EV(ARMV7, EVENT_F8H)		\
	__PMC_EV(ARMV7, EVENT_F9H)		\
	__PMC_EV(ARMV7, EVENT_FAH)		\
	__PMC_EV(ARMV7, EVENT_FBH)		\
	__PMC_EV(ARMV7, EVENT_FCH)		\
	__PMC_EV(ARMV7, EVENT_FDH)		\
	__PMC_EV(ARMV7, EVENT_FEH)		\
	__PMC_EV(ARMV7, EVENT_FFH)

#define	PMC_EV_ARMV7_FIRST	PMC_EV_ARMV7_EVENT_00H
#define	PMC_EV_ARMV7_LAST	PMC_EV_ARMV7_EVENT_FFH

#define	__PMC_EV_ALIAS_ARMV7_COMMON()					\
	__PMC_EV_ALIAS("PMNC_SW_INCR",		ARMV7_EVENT_00H)	\
	__PMC_EV_ALIAS("L1_ICACHE_REFILL",	ARMV7_EVENT_01H)	\
	__PMC_EV_ALIAS("ITLB_REFILL",		ARMV7_EVENT_02H)	\
	__PMC_EV_ALIAS("L1_DCACHE_REFILL",	ARMV7_EVENT_03H)	\
	__PMC_EV_ALIAS("L1_DCACHE_ACCESS",	ARMV7_EVENT_04H)	\
	__PMC_EV_ALIAS("DTLB_REFILL",		ARMV7_EVENT_05H)	\
	__PMC_EV_ALIAS("MEM_READ",		ARMV7_EVENT_06H)	\
	__PMC_EV_ALIAS("MEM_WRITE",		ARMV7_EVENT_07H)	\
	__PMC_EV_ALIAS("EXC_TAKEN",		ARMV7_EVENT_09H)	\
	__PMC_EV_ALIAS("EXC_EXECUTED",		ARMV7_EVENT_0AH)	\
	__PMC_EV_ALIAS("CID_WRITE",		ARMV7_EVENT_0BH)	\
	__PMC_EV_ALIAS("PC_WRITE",		ARMV7_EVENT_0CH)	\
	__PMC_EV_ALIAS("PC_IMM_BRANCH",		ARMV7_EVENT_0DH)	\
	__PMC_EV_ALIAS("MEM_UNALIGNED_ACCESS",	ARMV7_EVENT_0FH)	\
	__PMC_EV_ALIAS("PC_BRANCH_MIS_PRED",	ARMV7_EVENT_10H)	\
	__PMC_EV_ALIAS("CLOCK_CYCLES",		ARMV7_EVENT_11H)	\
	__PMC_EV_ALIAS("PC_BRANCH_PRED",	ARMV7_EVENT_12H)

#define	__PMC_EV_ALIAS_ARMV7_COMMON_A8()				\
	__PMC_EV_ALIAS_ARMV7_COMMON()					\
	__PMC_EV_ALIAS("INSTR_EXECUTED",	ARMV7_EVENT_08H)	\
	__PMC_EV_ALIAS("PC_PROC_RETURN",	ARMV7_EVENT_0EH)	\
	__PMC_EV_ALIAS("MEM_ACCESS",		ARMV7_EVENT_13H)	\
	__PMC_EV_ALIAS("L1_ICACHE_ACCESS",	ARMV7_EVENT_14H)	\
	__PMC_EV_ALIAS("L1_DCACHE_WB",		ARMV7_EVENT_15H)	\
	__PMC_EV_ALIAS("L2_CACHE_ACCESS",	ARMV7_EVENT_16H)	\
	__PMC_EV_ALIAS("L2_CACHE_REFILL",	ARMV7_EVENT_17H)	\
	__PMC_EV_ALIAS("L2_CACHE_WB",		ARMV7_EVENT_18H)	\
	__PMC_EV_ALIAS("BUS_ACCESS",		ARMV7_EVENT_19H)	\
	__PMC_EV_ALIAS("MEM_ERROR",		ARMV7_EVENT_1AH)	\
	__PMC_EV_ALIAS("INSTR_SPEC",		ARMV7_EVENT_1BH)	\
	__PMC_EV_ALIAS("TTBR_WRITE",		ARMV7_EVENT_1CH)	\
	__PMC_EV_ALIAS("BUS_CYCLES",		ARMV7_EVENT_1DH)	\
	__PMC_EV_ALIAS("CPU_CYCLES",		ARMV7_EVENT_FFH)

#define	__PMC_EV_ALIAS_ARMV7_CORTEX_A8()				 \
	__PMC_EV_ALIAS_ARMV7_COMMON_A8()				 \
	__PMC_EV_ALIAS("WRITE_BUF_FULL",		ARMV7_EVENT_40H) \
	__PMC_EV_ALIAS("L2_STORE_MERGED",		ARMV7_EVENT_41H) \
	__PMC_EV_ALIAS("L2_STORE_BUFFERABLE",		ARMV7_EVENT_42H) \
	__PMC_EV_ALIAS("L2_ACCESS",			ARMV7_EVENT_43H) \
	__PMC_EV_ALIAS("L2_CACHE_MISS",			ARMV7_EVENT_44H) \
	__PMC_EV_ALIAS("AXI_READ",			ARMV7_EVENT_45H) \
	__PMC_EV_ALIAS("AXI_WRITE",			ARMV7_EVENT_46H) \
	__PMC_EV_ALIAS("MEM_REPLAY_EVT",		ARMV7_EVENT_47H) \
	__PMC_EV_ALIAS("MEM_UNALIGNED_ACCESS_REPLAY",	ARMV7_EVENT_48H) \
	__PMC_EV_ALIAS("L1_DCACHE_HASH_MISS",		ARMV7_EVENT_49H) \
	__PMC_EV_ALIAS("L1_ICACHE_HASH_MISS",		ARMV7_EVENT_4AH) \
	__PMC_EV_ALIAS("L1_CACHE_PAGECOL_ALIAS",	ARMV7_EVENT_4BH) \
	__PMC_EV_ALIAS("L1_DCACHE_NEON_ACCESS",		ARMV7_EVENT_4CH) \
	__PMC_EV_ALIAS("L1_DCACHE_NEON_CACHEABLE",	ARMV7_EVENT_4DH) \
	__PMC_EV_ALIAS("L2_CACHE_NEON_MEM_ACCESS",	ARMV7_EVENT_4EH) \
	__PMC_EV_ALIAS("L2_CACHE_NEON_HIT",		ARMV7_EVENT_4FH) \
	__PMC_EV_ALIAS("L1_CACHE_ACCESS_NOCP15",	ARMV7_EVENT_50H) \
	__PMC_EV_ALIAS("RET_STACK_MISPREDICT",		ARMV7_EVENT_51H) \
	__PMC_EV_ALIAS("BRANCH_DIR_MISPREDICT",		ARMV7_EVENT_52H) \
	__PMC_EV_ALIAS("PRED_BRANCH_PRED_TAKEN",	ARMV7_EVENT_53H) \
	__PMC_EV_ALIAS("PRED_BRANCH_EXEC_TAKEN",	ARMV7_EVENT_54H) \
	__PMC_EV_ALIAS("OPS_ISSUED",			ARMV7_EVENT_55H) \
	__PMC_EV_ALIAS("CYCLES_NO_INSTRUCTION",		ARMV7_EVENT_56H) \
	__PMC_EV_ALIAS("INSTRUCTIONS_ISSUED_CYCLE",	ARMV7_EVENT_57H) \
	__PMC_EV_ALIAS("CYCLES_STALLED_NEON_MRC",	ARMV7_EVENT_58H) \
	__PMC_EV_ALIAS("CYCLES_STALLED_NEON_FULLQ",	ARMV7_EVENT_59H) \
	__PMC_EV_ALIAS("CYCLES_NONIDLE_NEON_INT",	ARMV7_EVENT_5AH) \
	__PMC_EV_ALIAS("PMUEXTIN0_EVT",			ARMV7_EVENT_70H) \
	__PMC_EV_ALIAS("PMUEXTIN1_EVT",			ARMV7_EVENT_71H) \
	__PMC_EV_ALIAS("PMUEXTIN_EVT",			ARMV7_EVENT_72H)
#define	PMC_EV_ARMV7_CORTEX_A8_FIRST	PMC_EV_ARMV7_PMNC_SW_INCR
#define	PMC_EV_ARMV7_CORTEX_A8_LAST	PMC_EV_ARMV7_PMUEXTIN_EVT

#define	__PMC_EV_ALIAS_ARMV7_CORTEX_A9()					\
	__PMC_EV_ALIAS_ARMV7_COMMON()						\
	__PMC_EV_ALIAS("JAVA_BYTECODE",			ARMV7_EVENT_40H)	\
	__PMC_EV_ALIAS("SOFTWARE_JAVA_BYTECODE",	ARMV7_EVENT_41H)	\
	__PMC_EV_ALIAS("JAZELLE_BACKWARD_BRANCH",	ARMV7_EVENT_42H)	\
	__PMC_EV_ALIAS("COHERENT_LINEFILL_MISSC",	ARMV7_EVENT_50H)	\
	__PMC_EV_ALIAS("COHERENT_LINEFILL_HITC",	ARMV7_EVENT_51H)	\
	__PMC_EV_ALIAS("INSTR_CACHE_DEPENDENT_STALL",	ARMV7_EVENT_60H)	\
	__PMC_EV_ALIAS("DATA_CACHE_DEPENDENT_STALL",	ARMV7_EVENT_61H)	\
	__PMC_EV_ALIAS("MAIN_TLB_MISS_STALL",		ARMV7_EVENT_62H)	\
	__PMC_EV_ALIAS("STREX_PASSED",			ARMV7_EVENT_63H)	\
	__PMC_EV_ALIAS("STREX_FAILED",			ARMV7_EVENT_64H)	\
	__PMC_EV_ALIAS("DATA_EVICTION",			ARMV7_EVENT_65H)	\
	__PMC_EV_ALIAS("ISSUE_DNOT_DISPATCH_ANY_INSTR",	ARMV7_EVENT_66H)	\
	__PMC_EV_ALIAS("ISSUE_IS_EMPTY",		ARMV7_EVENT_67H)	\
	__PMC_EV_ALIAS("INSTR_RENAMED",			ARMV7_EVENT_68H)	\
	__PMC_EV_ALIAS("PREDICTABLE_FUNCTION_RETURN",	ARMV7_EVENT_6EH)	\
	__PMC_EV_ALIAS("MAIN_EXECUTION_UNIT_PIPE",	ARMV7_EVENT_70H)	\
	__PMC_EV_ALIAS("SECOND_EXECUTION_UNIT_PIPE",	ARMV7_EVENT_71H)	\
	__PMC_EV_ALIAS("LOAD_STORE_PIPE",		ARMV7_EVENT_72H)	\
	__PMC_EV_ALIAS("FLOATING_POINT_INSTR_RENAMED",	ARMV7_EVENT_73H)	\
	__PMC_EV_ALIAS("NEON_INSTRS_RENAMED",		ARMV7_EVENT_74H)	\
	__PMC_EV_ALIAS("PLD_STALL",			ARMV7_EVENT_80H)	\
	__PMC_EV_ALIAS("WRITE_STALL",			ARMV7_EVENT_81H)	\
	__PMC_EV_ALIAS("INSTR_MAIN_TLB_MISS_STALL",	ARMV7_EVENT_82H)	\
	__PMC_EV_ALIAS("DATA_MAIN_TLB_MISS_STALL",	ARMV7_EVENT_83H)	\
	__PMC_EV_ALIAS("INSTR_MICRO_TLB_MISS_STALL",	ARMV7_EVENT_84H)	\
	__PMC_EV_ALIAS("DATA_MICRO_TLB_MISS_STALL",	ARMV7_EVENT_85H)	\
	__PMC_EV_ALIAS("DMB_STALL",			ARMV7_EVENT_86H)	\
	__PMC_EV_ALIAS("INTEGER_CORE_CLOCK_ENABLED",	ARMV7_EVENT_8AH)	\
	__PMC_EV_ALIAS("DATA_ENGINE_CLOCK_ENABLED",	ARMV7_EVENT_8BH)	\
	__PMC_EV_ALIAS("ISB",				ARMV7_EVENT_90H)	\
	__PMC_EV_ALIAS("DSB",				ARMV7_EVENT_91H)	\
	__PMC_EV_ALIAS("DMB",				ARMV7_EVENT_92H)	\
	__PMC_EV_ALIAS("EXTERNAL_INTERRUPT",		ARMV7_EVENT_93H)	\
	__PMC_EV_ALIAS("PLE_CACHE_LINE_REQ_COMPLETED",	ARMV7_EVENT_A0H)	\
	__PMC_EV_ALIAS("PLE_CACHE_LINE_REQ_SKIPPED",	ARMV7_EVENT_A1H)	\
	__PMC_EV_ALIAS("PLE_FIFO_FLUSH",		ARMV7_EVENT_A2H)	\
	__PMC_EV_ALIAS("PLE_REQUEST_COMPLETED",		ARMV7_EVENT_A3H)	\
	__PMC_EV_ALIAS("PLE_FIFO_OVERFLOW",		ARMV7_EVENT_A4H)	\
	__PMC_EV_ALIAS("PLE_REQUEST_PROGRAMMED",	ARMV7_EVENT_A5H)

/*
 * ARMv8 Events
 *
 * NB: ARMv8.1 extends this to a 16-bit encoding, but we only define the 10-bit
 * event numbers here.
 */

#define	__PMC_EV_ARMV8()			\
	__PMC_EV(ARMV8, EVENT_00H)		\
	__PMC_EV(ARMV8, EVENT_01H)		\
	__PMC_EV(ARMV8, EVENT_02H)		\
	__PMC_EV(ARMV8, EVENT_03H)		\
	__PMC_EV(ARMV8, EVENT_04H)		\
	__PMC_EV(ARMV8, EVENT_05H)		\
	__PMC_EV(ARMV8, EVENT_06H)		\
	__PMC_EV(ARMV8, EVENT_07H)		\
	__PMC_EV(ARMV8, EVENT_08H)		\
	__PMC_EV(ARMV8, EVENT_09H)		\
	__PMC_EV(ARMV8, EVENT_0AH)		\
	__PMC_EV(ARMV8, EVENT_0BH)		\
	__PMC_EV(ARMV8, EVENT_0CH)		\
	__PMC_EV(ARMV8, EVENT_0DH)		\
	__PMC_EV(ARMV8, EVENT_0EH)		\
	__PMC_EV(ARMV8, EVENT_0FH)		\
	__PMC_EV(ARMV8, EVENT_10H)		\
	__PMC_EV(ARMV8, EVENT_11H)		\
	__PMC_EV(ARMV8, EVENT_12H)		\
	__PMC_EV(ARMV8, EVENT_13H)		\
	__PMC_EV(ARMV8, EVENT_14H)		\
	__PMC_EV(ARMV8, EVENT_15H)		\
	__PMC_EV(ARMV8, EVENT_16H)		\
	__PMC_EV(ARMV8, EVENT_17H)		\
	__PMC_EV(ARMV8, EVENT_18H)		\
	__PMC_EV(ARMV8, EVENT_19H)		\
	__PMC_EV(ARMV8, EVENT_1AH)		\
	__PMC_EV(ARMV8, EVENT_1BH)		\
	__PMC_EV(ARMV8, EVENT_1CH)		\
	__PMC_EV(ARMV8, EVENT_1DH)		\
	__PMC_EV(ARMV8, EVENT_1EH)		\
	__PMC_EV(ARMV8, EVENT_1FH)		\
	__PMC_EV(ARMV8, EVENT_20H)		\
	__PMC_EV(ARMV8, EVENT_21H)		\
	__PMC_EV(ARMV8, EVENT_22H)		\
	__PMC_EV(ARMV8, EVENT_23H)		\
	__PMC_EV(ARMV8, EVENT_24H)		\
	__PMC_EV(ARMV8, EVENT_25H)		\
	__PMC_EV(ARMV8, EVENT_26H)		\
	__PMC_EV(ARMV8, EVENT_27H)		\
	__PMC_EV(ARMV8, EVENT_28H)		\
	__PMC_EV(ARMV8, EVENT_29H)		\
	__PMC_EV(ARMV8, EVENT_2AH)		\
	__PMC_EV(ARMV8, EVENT_2BH)		\
	__PMC_EV(ARMV8, EVENT_2CH)		\
	__PMC_EV(ARMV8, EVENT_2DH)		\
	__PMC_EV(ARMV8, EVENT_2EH)		\
	__PMC_EV(ARMV8, EVENT_2FH)		\
	__PMC_EV(ARMV8, EVENT_30H)		\
	__PMC_EV(ARMV8, EVENT_31H)		\
	__PMC_EV(ARMV8, EVENT_32H)		\
	__PMC_EV(ARMV8, EVENT_33H)		\
	__PMC_EV(ARMV8, EVENT_34H)		\
	__PMC_EV(ARMV8, EVENT_35H)		\
	__PMC_EV(ARMV8, EVENT_36H)		\
	__PMC_EV(ARMV8, EVENT_37H)		\
	__PMC_EV(ARMV8, EVENT_38H)		\
	__PMC_EV(ARMV8, EVENT_39H)		\
	__PMC_EV(ARMV8, EVENT_3AH)		\
	__PMC_EV(ARMV8, EVENT_3BH)		\
	__PMC_EV(ARMV8, EVENT_3CH)		\
	__PMC_EV(ARMV8, EVENT_3DH)		\
	__PMC_EV(ARMV8, EVENT_3EH)		\
	__PMC_EV(ARMV8, EVENT_3FH)		\
	__PMC_EV(ARMV8, EVENT_40H)		\
	__PMC_EV(ARMV8, EVENT_41H)		\
	__PMC_EV(ARMV8, EVENT_42H)		\
	__PMC_EV(ARMV8, EVENT_43H)		\
	__PMC_EV(ARMV8, EVENT_44H)		\
	__PMC_EV(ARMV8, EVENT_45H)		\
	__PMC_EV(ARMV8, EVENT_46H)		\
	__PMC_EV(ARMV8, EVENT_47H)		\
	__PMC_EV(ARMV8, EVENT_48H)		\
	__PMC_EV(ARMV8, EVENT_49H)		\
	__PMC_EV(ARMV8, EVENT_4AH)		\
	__PMC_EV(ARMV8, EVENT_4BH)		\
	__PMC_EV(ARMV8, EVENT_4CH)		\
	__PMC_EV(ARMV8, EVENT_4DH)		\
	__PMC_EV(ARMV8, EVENT_4EH)		\
	__PMC_EV(ARMV8, EVENT_4FH)		\
	__PMC_EV(ARMV8, EVENT_50H)		\
	__PMC_EV(ARMV8, EVENT_51H)		\
	__PMC_EV(ARMV8, EVENT_52H)		\
	__PMC_EV(ARMV8, EVENT_53H)		\
	__PMC_EV(ARMV8, EVENT_54H)		\
	__PMC_EV(ARMV8, EVENT_55H)		\
	__PMC_EV(ARMV8, EVENT_56H)		\
	__PMC_EV(ARMV8, EVENT_57H)		\
	__PMC_EV(ARMV8, EVENT_58H)		\
	__PMC_EV(ARMV8, EVENT_59H)		\
	__PMC_EV(ARMV8, EVENT_5AH)		\
	__PMC_EV(ARMV8, EVENT_5BH)		\
	__PMC_EV(ARMV8, EVENT_5CH)		\
	__PMC_EV(ARMV8, EVENT_5DH)		\
	__PMC_EV(ARMV8, EVENT_5EH)		\
	__PMC_EV(ARMV8, EVENT_5FH)		\
	__PMC_EV(ARMV8, EVENT_60H)		\
	__PMC_EV(ARMV8, EVENT_61H)		\
	__PMC_EV(ARMV8, EVENT_62H)		\
	__PMC_EV(ARMV8, EVENT_63H)		\
	__PMC_EV(ARMV8, EVENT_64H)		\
	__PMC_EV(ARMV8, EVENT_65H)		\
	__PMC_EV(ARMV8, EVENT_66H)		\
	__PMC_EV(ARMV8, EVENT_67H)		\
	__PMC_EV(ARMV8, EVENT_68H)		\
	__PMC_EV(ARMV8, EVENT_69H)		\
	__PMC_EV(ARMV8, EVENT_6AH)		\
	__PMC_EV(ARMV8, EVENT_6BH)		\
	__PMC_EV(ARMV8, EVENT_6CH)		\
	__PMC_EV(ARMV8, EVENT_6DH)		\
	__PMC_EV(ARMV8, EVENT_6EH)		\
	__PMC_EV(ARMV8, EVENT_6FH)		\
	__PMC_EV(ARMV8, EVENT_70H)		\
	__PMC_EV(ARMV8, EVENT_71H)		\
	__PMC_EV(ARMV8, EVENT_72H)		\
	__PMC_EV(ARMV8, EVENT_73H)		\
	__PMC_EV(ARMV8, EVENT_74H)		\
	__PMC_EV(ARMV8, EVENT_75H)		\
	__PMC_EV(ARMV8, EVENT_76H)		\
	__PMC_EV(ARMV8, EVENT_77H)		\
	__PMC_EV(ARMV8, EVENT_78H)		\
	__PMC_EV(ARMV8, EVENT_79H)		\
	__PMC_EV(ARMV8, EVENT_7AH)		\
	__PMC_EV(ARMV8, EVENT_7BH)		\
	__PMC_EV(ARMV8, EVENT_7CH)		\
	__PMC_EV(ARMV8, EVENT_7DH)		\
	__PMC_EV(ARMV8, EVENT_7EH)		\
	__PMC_EV(ARMV8, EVENT_7FH)		\
	__PMC_EV(ARMV8, EVENT_80H)		\
	__PMC_EV(ARMV8, EVENT_81H)		\
	__PMC_EV(ARMV8, EVENT_82H)		\
	__PMC_EV(ARMV8, EVENT_83H)		\
	__PMC_EV(ARMV8, EVENT_84H)		\
	__PMC_EV(ARMV8, EVENT_85H)		\
	__PMC_EV(ARMV8, EVENT_86H)		\
	__PMC_EV(ARMV8, EVENT_87H)		\
	__PMC_EV(ARMV8, EVENT_88H)		\
	__PMC_EV(ARMV8, EVENT_89H)		\
	__PMC_EV(ARMV8, EVENT_8AH)		\
	__PMC_EV(ARMV8, EVENT_8BH)		\
	__PMC_EV(ARMV8, EVENT_8CH)		\
	__PMC_EV(ARMV8, EVENT_8DH)		\
	__PMC_EV(ARMV8, EVENT_8EH)		\
	__PMC_EV(ARMV8, EVENT_8FH)		\
	__PMC_EV(ARMV8, EVENT_90H)		\
	__PMC_EV(ARMV8, EVENT_91H)		\
	__PMC_EV(ARMV8, EVENT_92H)		\
	__PMC_EV(ARMV8, EVENT_93H)		\
	__PMC_EV(ARMV8, EVENT_94H)		\
	__PMC_EV(ARMV8, EVENT_95H)		\
	__PMC_EV(ARMV8, EVENT_96H)		\
	__PMC_EV(ARMV8, EVENT_97H)		\
	__PMC_EV(ARMV8, EVENT_98H)		\
	__PMC_EV(ARMV8, EVENT_99H)		\
	__PMC_EV(ARMV8, EVENT_9AH)		\
	__PMC_EV(ARMV8, EVENT_9BH)		\
	__PMC_EV(ARMV8, EVENT_9CH)		\
	__PMC_EV(ARMV8, EVENT_9DH)		\
	__PMC_EV(ARMV8, EVENT_9EH)		\
	__PMC_EV(ARMV8, EVENT_9FH)		\
	__PMC_EV(ARMV8, EVENT_A0H)		\
	__PMC_EV(ARMV8, EVENT_A1H)		\
	__PMC_EV(ARMV8, EVENT_A2H)		\
	__PMC_EV(ARMV8, EVENT_A3H)		\
	__PMC_EV(ARMV8, EVENT_A4H)		\
	__PMC_EV(ARMV8, EVENT_A5H)		\
	__PMC_EV(ARMV8, EVENT_A6H)		\
	__PMC_EV(ARMV8, EVENT_A7H)		\
	__PMC_EV(ARMV8, EVENT_A8H)		\
	__PMC_EV(ARMV8, EVENT_A9H)		\
	__PMC_EV(ARMV8, EVENT_AAH)		\
	__PMC_EV(ARMV8, EVENT_ABH)		\
	__PMC_EV(ARMV8, EVENT_ACH)		\
	__PMC_EV(ARMV8, EVENT_ADH)		\
	__PMC_EV(ARMV8, EVENT_AEH)		\
	__PMC_EV(ARMV8, EVENT_AFH)		\
	__PMC_EV(ARMV8, EVENT_B0H)		\
	__PMC_EV(ARMV8, EVENT_B1H)		\
	__PMC_EV(ARMV8, EVENT_B2H)		\
	__PMC_EV(ARMV8, EVENT_B3H)		\
	__PMC_EV(ARMV8, EVENT_B4H)		\
	__PMC_EV(ARMV8, EVENT_B5H)		\
	__PMC_EV(ARMV8, EVENT_B6H)		\
	__PMC_EV(ARMV8, EVENT_B7H)		\
	__PMC_EV(ARMV8, EVENT_B8H)		\
	__PMC_EV(ARMV8, EVENT_B9H)		\
	__PMC_EV(ARMV8, EVENT_BAH)		\
	__PMC_EV(ARMV8, EVENT_BBH)		\
	__PMC_EV(ARMV8, EVENT_BCH)		\
	__PMC_EV(ARMV8, EVENT_BDH)		\
	__PMC_EV(ARMV8, EVENT_BEH)		\
	__PMC_EV(ARMV8, EVENT_BFH)		\
	__PMC_EV(ARMV8, EVENT_C0H)		\
	__PMC_EV(ARMV8, EVENT_C1H)		\
	__PMC_EV(ARMV8, EVENT_C2H)		\
	__PMC_EV(ARMV8, EVENT_C3H)		\
	__PMC_EV(ARMV8, EVENT_C4H)		\
	__PMC_EV(ARMV8, EVENT_C5H)		\
	__PMC_EV(ARMV8, EVENT_C6H)		\
	__PMC_EV(ARMV8, EVENT_C7H)		\
	__PMC_EV(ARMV8, EVENT_C8H)		\
	__PMC_EV(ARMV8, EVENT_C9H)		\
	__PMC_EV(ARMV8, EVENT_CAH)		\
	__PMC_EV(ARMV8, EVENT_CBH)		\
	__PMC_EV(ARMV8, EVENT_CCH)		\
	__PMC_EV(ARMV8, EVENT_CDH)		\
	__PMC_EV(ARMV8, EVENT_CEH)		\
	__PMC_EV(ARMV8, EVENT_CFH)		\
	__PMC_EV(ARMV8, EVENT_D0H)		\
	__PMC_EV(ARMV8, EVENT_D1H)		\
	__PMC_EV(ARMV8, EVENT_D2H)		\
	__PMC_EV(ARMV8, EVENT_D3H)		\
	__PMC_EV(ARMV8, EVENT_D4H)		\
	__PMC_EV(ARMV8, EVENT_D5H)		\
	__PMC_EV(ARMV8, EVENT_D6H)		\
	__PMC_EV(ARMV8, EVENT_D7H)		\
	__PMC_EV(ARMV8, EVENT_D8H)		\
	__PMC_EV(ARMV8, EVENT_D9H)		\
	__PMC_EV(ARMV8, EVENT_DAH)		\
	__PMC_EV(ARMV8, EVENT_DBH)		\
	__PMC_EV(ARMV8, EVENT_DCH)		\
	__PMC_EV(ARMV8, EVENT_DDH)		\
	__PMC_EV(ARMV8, EVENT_DEH)		\
	__PMC_EV(ARMV8, EVENT_DFH)		\
	__PMC_EV(ARMV8, EVENT_E0H)		\
	__PMC_EV(ARMV8, EVENT_E1H)		\
	__PMC_EV(ARMV8, EVENT_E2H)		\
	__PMC_EV(ARMV8, EVENT_E3H)		\
	__PMC_EV(ARMV8, EVENT_E4H)		\
	__PMC_EV(ARMV8, EVENT_E5H)		\
	__PMC_EV(ARMV8, EVENT_E6H)		\
	__PMC_EV(ARMV8, EVENT_E7H)		\
	__PMC_EV(ARMV8, EVENT_E8H)		\
	__PMC_EV(ARMV8, EVENT_E9H)		\
	__PMC_EV(ARMV8, EVENT_EAH)		\
	__PMC_EV(ARMV8, EVENT_EBH)		\
	__PMC_EV(ARMV8, EVENT_ECH)		\
	__PMC_EV(ARMV8, EVENT_EDH)		\
	__PMC_EV(ARMV8, EVENT_EEH)		\
	__PMC_EV(ARMV8, EVENT_EFH)		\
	__PMC_EV(ARMV8, EVENT_F0H)		\
	__PMC_EV(ARMV8, EVENT_F1H)		\
	__PMC_EV(ARMV8, EVENT_F2H)		\
	__PMC_EV(ARMV8, EVENT_F3H)		\
	__PMC_EV(ARMV8, EVENT_F4H)		\
	__PMC_EV(ARMV8, EVENT_F5H)		\
	__PMC_EV(ARMV8, EVENT_F6H)		\
	__PMC_EV(ARMV8, EVENT_F7H)		\
	__PMC_EV(ARMV8, EVENT_F8H)		\
	__PMC_EV(ARMV8, EVENT_F9H)		\
	__PMC_EV(ARMV8, EVENT_FAH)		\
	__PMC_EV(ARMV8, EVENT_FBH)		\
	__PMC_EV(ARMV8, EVENT_FCH)		\
	__PMC_EV(ARMV8, EVENT_FDH)		\
	__PMC_EV(ARMV8, EVENT_FEH)		\
	__PMC_EV(ARMV8, EVENT_FFH)		\
	__PMC_EV(ARMV8, EVENT_100H)		\
	__PMC_EV(ARMV8, EVENT_101H)		\
	__PMC_EV(ARMV8, EVENT_102H)		\
	__PMC_EV(ARMV8, EVENT_103H)		\
	__PMC_EV(ARMV8, EVENT_104H)		\
	__PMC_EV(ARMV8, EVENT_105H)		\
	__PMC_EV(ARMV8, EVENT_106H)		\
	__PMC_EV(ARMV8, EVENT_107H)		\
	__PMC_EV(ARMV8, EVENT_108H)		\
	__PMC_EV(ARMV8, EVENT_109H)		\
	__PMC_EV(ARMV8, EVENT_10AH)		\
	__PMC_EV(ARMV8, EVENT_10BH)		\
	__PMC_EV(ARMV8, EVENT_10CH)		\
	__PMC_EV(ARMV8, EVENT_10DH)		\
	__PMC_EV(ARMV8, EVENT_10EH)		\
	__PMC_EV(ARMV8, EVENT_10FH)		\
	__PMC_EV(ARMV8, EVENT_110H)		\
	__PMC_EV(ARMV8, EVENT_111H)		\
	__PMC_EV(ARMV8, EVENT_112H)		\
	__PMC_EV(ARMV8, EVENT_113H)		\
	__PMC_EV(ARMV8, EVENT_114H)		\
	__PMC_EV(ARMV8, EVENT_115H)		\
	__PMC_EV(ARMV8, EVENT_116H)		\
	__PMC_EV(ARMV8, EVENT_117H)		\
	__PMC_EV(ARMV8, EVENT_118H)		\
	__PMC_EV(ARMV8, EVENT_119H)		\
	__PMC_EV(ARMV8, EVENT_11AH)		\
	__PMC_EV(ARMV8, EVENT_11BH)		\
	__PMC_EV(ARMV8, EVENT_11CH)		\
	__PMC_EV(ARMV8, EVENT_11DH)		\
	__PMC_EV(ARMV8, EVENT_11EH)		\
	__PMC_EV(ARMV8, EVENT_11FH)		\
	__PMC_EV(ARMV8, EVENT_120H)		\
	__PMC_EV(ARMV8, EVENT_121H)		\
	__PMC_EV(ARMV8, EVENT_122H)		\
	__PMC_EV(ARMV8, EVENT_123H)		\
	__PMC_EV(ARMV8, EVENT_124H)		\
	__PMC_EV(ARMV8, EVENT_125H)		\
	__PMC_EV(ARMV8, EVENT_126H)		\
	__PMC_EV(ARMV8, EVENT_127H)		\
	__PMC_EV(ARMV8, EVENT_128H)		\
	__PMC_EV(ARMV8, EVENT_129H)		\
	__PMC_EV(ARMV8, EVENT_12AH)		\
	__PMC_EV(ARMV8, EVENT_12BH)		\
	__PMC_EV(ARMV8, EVENT_12CH)		\
	__PMC_EV(ARMV8, EVENT_12DH)		\
	__PMC_EV(ARMV8, EVENT_12EH)		\
	__PMC_EV(ARMV8, EVENT_12FH)		\
	__PMC_EV(ARMV8, EVENT_130H)		\
	__PMC_EV(ARMV8, EVENT_131H)		\
	__PMC_EV(ARMV8, EVENT_132H)		\
	__PMC_EV(ARMV8, EVENT_133H)		\
	__PMC_EV(ARMV8, EVENT_134H)		\
	__PMC_EV(ARMV8, EVENT_135H)		\
	__PMC_EV(ARMV8, EVENT_136H)		\
	__PMC_EV(ARMV8, EVENT_137H)		\
	__PMC_EV(ARMV8, EVENT_138H)		\
	__PMC_EV(ARMV8, EVENT_139H)		\
	__PMC_EV(ARMV8, EVENT_13AH)		\
	__PMC_EV(ARMV8, EVENT_13BH)		\
	__PMC_EV(ARMV8, EVENT_13CH)		\
	__PMC_EV(ARMV8, EVENT_13DH)		\
	__PMC_EV(ARMV8, EVENT_13EH)		\
	__PMC_EV(ARMV8, EVENT_13FH)		\
	__PMC_EV(ARMV8, EVENT_140H)		\
	__PMC_EV(ARMV8, EVENT_141H)		\
	__PMC_EV(ARMV8, EVENT_142H)		\
	__PMC_EV(ARMV8, EVENT_143H)		\
	__PMC_EV(ARMV8, EVENT_144H)		\
	__PMC_EV(ARMV8, EVENT_145H)		\
	__PMC_EV(ARMV8, EVENT_146H)		\
	__PMC_EV(ARMV8, EVENT_147H)		\
	__PMC_EV(ARMV8, EVENT_148H)		\
	__PMC_EV(ARMV8, EVENT_149H)		\
	__PMC_EV(ARMV8, EVENT_14AH)		\
	__PMC_EV(ARMV8, EVENT_14BH)		\
	__PMC_EV(ARMV8, EVENT_14CH)		\
	__PMC_EV(ARMV8, EVENT_14DH)		\
	__PMC_EV(ARMV8, EVENT_14EH)		\
	__PMC_EV(ARMV8, EVENT_14FH)		\
	__PMC_EV(ARMV8, EVENT_150H)		\
	__PMC_EV(ARMV8, EVENT_151H)		\
	__PMC_EV(ARMV8, EVENT_152H)		\
	__PMC_EV(ARMV8, EVENT_153H)		\
	__PMC_EV(ARMV8, EVENT_154H)		\
	__PMC_EV(ARMV8, EVENT_155H)		\
	__PMC_EV(ARMV8, EVENT_156H)		\
	__PMC_EV(ARMV8, EVENT_157H)		\
	__PMC_EV(ARMV8, EVENT_158H)		\
	__PMC_EV(ARMV8, EVENT_159H)		\
	__PMC_EV(ARMV8, EVENT_15AH)		\
	__PMC_EV(ARMV8, EVENT_15BH)		\
	__PMC_EV(ARMV8, EVENT_15CH)		\
	__PMC_EV(ARMV8, EVENT_15DH)		\
	__PMC_EV(ARMV8, EVENT_15EH)		\
	__PMC_EV(ARMV8, EVENT_15FH)		\
	__PMC_EV(ARMV8, EVENT_160H)		\
	__PMC_EV(ARMV8, EVENT_161H)		\
	__PMC_EV(ARMV8, EVENT_162H)		\
	__PMC_EV(ARMV8, EVENT_163H)		\
	__PMC_EV(ARMV8, EVENT_164H)		\
	__PMC_EV(ARMV8, EVENT_165H)		\
	__PMC_EV(ARMV8, EVENT_166H)		\
	__PMC_EV(ARMV8, EVENT_167H)		\
	__PMC_EV(ARMV8, EVENT_168H)		\
	__PMC_EV(ARMV8, EVENT_169H)		\
	__PMC_EV(ARMV8, EVENT_16AH)		\
	__PMC_EV(ARMV8, EVENT_16BH)		\
	__PMC_EV(ARMV8, EVENT_16CH)		\
	__PMC_EV(ARMV8, EVENT_16DH)		\
	__PMC_EV(ARMV8, EVENT_16EH)		\
	__PMC_EV(ARMV8, EVENT_16FH)		\
	__PMC_EV(ARMV8, EVENT_170H)		\
	__PMC_EV(ARMV8, EVENT_171H)		\
	__PMC_EV(ARMV8, EVENT_172H)		\
	__PMC_EV(ARMV8, EVENT_173H)		\
	__PMC_EV(ARMV8, EVENT_174H)		\
	__PMC_EV(ARMV8, EVENT_175H)		\
	__PMC_EV(ARMV8, EVENT_176H)		\
	__PMC_EV(ARMV8, EVENT_177H)		\
	__PMC_EV(ARMV8, EVENT_178H)		\
	__PMC_EV(ARMV8, EVENT_179H)		\
	__PMC_EV(ARMV8, EVENT_17AH)		\
	__PMC_EV(ARMV8, EVENT_17BH)		\
	__PMC_EV(ARMV8, EVENT_17CH)		\
	__PMC_EV(ARMV8, EVENT_17DH)		\
	__PMC_EV(ARMV8, EVENT_17EH)		\
	__PMC_EV(ARMV8, EVENT_17FH)		\
	__PMC_EV(ARMV8, EVENT_180H)		\
	__PMC_EV(ARMV8, EVENT_181H)		\
	__PMC_EV(ARMV8, EVENT_182H)		\
	__PMC_EV(ARMV8, EVENT_183H)		\
	__PMC_EV(ARMV8, EVENT_184H)		\
	__PMC_EV(ARMV8, EVENT_185H)		\
	__PMC_EV(ARMV8, EVENT_186H)		\
	__PMC_EV(ARMV8, EVENT_187H)		\
	__PMC_EV(ARMV8, EVENT_188H)		\
	__PMC_EV(ARMV8, EVENT_189H)		\
	__PMC_EV(ARMV8, EVENT_18AH)		\
	__PMC_EV(ARMV8, EVENT_18BH)		\
	__PMC_EV(ARMV8, EVENT_18CH)		\
	__PMC_EV(ARMV8, EVENT_18DH)		\
	__PMC_EV(ARMV8, EVENT_18EH)		\
	__PMC_EV(ARMV8, EVENT_18FH)		\
	__PMC_EV(ARMV8, EVENT_190H)		\
	__PMC_EV(ARMV8, EVENT_191H)		\
	__PMC_EV(ARMV8, EVENT_192H)		\
	__PMC_EV(ARMV8, EVENT_193H)		\
	__PMC_EV(ARMV8, EVENT_194H)		\
	__PMC_EV(ARMV8, EVENT_195H)		\
	__PMC_EV(ARMV8, EVENT_196H)		\
	__PMC_EV(ARMV8, EVENT_197H)		\
	__PMC_EV(ARMV8, EVENT_198H)		\
	__PMC_EV(ARMV8, EVENT_199H)		\
	__PMC_EV(ARMV8, EVENT_19AH)		\
	__PMC_EV(ARMV8, EVENT_19BH)		\
	__PMC_EV(ARMV8, EVENT_19CH)		\
	__PMC_EV(ARMV8, EVENT_19DH)		\
	__PMC_EV(ARMV8, EVENT_19EH)		\
	__PMC_EV(ARMV8, EVENT_19FH)		\
	__PMC_EV(ARMV8, EVENT_1A0H)		\
	__PMC_EV(ARMV8, EVENT_1A1H)		\
	__PMC_EV(ARMV8, EVENT_1A2H)		\
	__PMC_EV(ARMV8, EVENT_1A3H)		\
	__PMC_EV(ARMV8, EVENT_1A4H)		\
	__PMC_EV(ARMV8, EVENT_1A5H)		\
	__PMC_EV(ARMV8, EVENT_1A6H)		\
	__PMC_EV(ARMV8, EVENT_1A7H)		\
	__PMC_EV(ARMV8, EVENT_1A8H)		\
	__PMC_EV(ARMV8, EVENT_1A9H)		\
	__PMC_EV(ARMV8, EVENT_1AAH)		\
	__PMC_EV(ARMV8, EVENT_1ABH)		\
	__PMC_EV(ARMV8, EVENT_1ACH)		\
	__PMC_EV(ARMV8, EVENT_1ADH)		\
	__PMC_EV(ARMV8, EVENT_1AEH)		\
	__PMC_EV(ARMV8, EVENT_1AFH)		\
	__PMC_EV(ARMV8, EVENT_1B0H)		\
	__PMC_EV(ARMV8, EVENT_1B1H)		\
	__PMC_EV(ARMV8, EVENT_1B2H)		\
	__PMC_EV(ARMV8, EVENT_1B3H)		\
	__PMC_EV(ARMV8, EVENT_1B4H)		\
	__PMC_EV(ARMV8, EVENT_1B5H)		\
	__PMC_EV(ARMV8, EVENT_1B6H)		\
	__PMC_EV(ARMV8, EVENT_1B7H)		\
	__PMC_EV(ARMV8, EVENT_1B8H)		\
	__PMC_EV(ARMV8, EVENT_1B9H)		\
	__PMC_EV(ARMV8, EVENT_1BAH)		\
	__PMC_EV(ARMV8, EVENT_1BBH)		\
	__PMC_EV(ARMV8, EVENT_1BCH)		\
	__PMC_EV(ARMV8, EVENT_1BDH)		\
	__PMC_EV(ARMV8, EVENT_1BEH)		\
	__PMC_EV(ARMV8, EVENT_1BFH)		\
	__PMC_EV(ARMV8, EVENT_1C0H)		\
	__PMC_EV(ARMV8, EVENT_1C1H)		\
	__PMC_EV(ARMV8, EVENT_1C2H)		\
	__PMC_EV(ARMV8, EVENT_1C3H)		\
	__PMC_EV(ARMV8, EVENT_1C4H)		\
	__PMC_EV(ARMV8, EVENT_1C5H)		\
	__PMC_EV(ARMV8, EVENT_1C6H)		\
	__PMC_EV(ARMV8, EVENT_1C7H)		\
	__PMC_EV(ARMV8, EVENT_1C8H)		\
	__PMC_EV(ARMV8, EVENT_1C9H)		\
	__PMC_EV(ARMV8, EVENT_1CAH)		\
	__PMC_EV(ARMV8, EVENT_1CBH)		\
	__PMC_EV(ARMV8, EVENT_1CCH)		\
	__PMC_EV(ARMV8, EVENT_1CDH)		\
	__PMC_EV(ARMV8, EVENT_1CEH)		\
	__PMC_EV(ARMV8, EVENT_1CFH)		\
	__PMC_EV(ARMV8, EVENT_1D0H)		\
	__PMC_EV(ARMV8, EVENT_1D1H)		\
	__PMC_EV(ARMV8, EVENT_1D2H)		\
	__PMC_EV(ARMV8, EVENT_1D3H)		\
	__PMC_EV(ARMV8, EVENT_1D4H)		\
	__PMC_EV(ARMV8, EVENT_1D5H)		\
	__PMC_EV(ARMV8, EVENT_1D6H)		\
	__PMC_EV(ARMV8, EVENT_1D7H)		\
	__PMC_EV(ARMV8, EVENT_1D8H)		\
	__PMC_EV(ARMV8, EVENT_1D9H)		\
	__PMC_EV(ARMV8, EVENT_1DAH)		\
	__PMC_EV(ARMV8, EVENT_1DBH)		\
	__PMC_EV(ARMV8, EVENT_1DCH)		\
	__PMC_EV(ARMV8, EVENT_1DDH)		\
	__PMC_EV(ARMV8, EVENT_1DEH)		\
	__PMC_EV(ARMV8, EVENT_1DFH)		\
	__PMC_EV(ARMV8, EVENT_1E0H)		\
	__PMC_EV(ARMV8, EVENT_1E1H)		\
	__PMC_EV(ARMV8, EVENT_1E2H)		\
	__PMC_EV(ARMV8, EVENT_1E3H)		\
	__PMC_EV(ARMV8, EVENT_1E4H)		\
	__PMC_EV(ARMV8, EVENT_1E5H)		\
	__PMC_EV(ARMV8, EVENT_1E6H)		\
	__PMC_EV(ARMV8, EVENT_1E7H)		\
	__PMC_EV(ARMV8, EVENT_1E8H)		\
	__PMC_EV(ARMV8, EVENT_1E9H)		\
	__PMC_EV(ARMV8, EVENT_1EAH)		\
	__PMC_EV(ARMV8, EVENT_1EBH)		\
	__PMC_EV(ARMV8, EVENT_1ECH)		\
	__PMC_EV(ARMV8, EVENT_1EDH)		\
	__PMC_EV(ARMV8, EVENT_1EEH)		\
	__PMC_EV(ARMV8, EVENT_1EFH)		\
	__PMC_EV(ARMV8, EVENT_1F0H)		\
	__PMC_EV(ARMV8, EVENT_1F1H)		\
	__PMC_EV(ARMV8, EVENT_1F2H)		\
	__PMC_EV(ARMV8, EVENT_1F3H)		\
	__PMC_EV(ARMV8, EVENT_1F4H)		\
	__PMC_EV(ARMV8, EVENT_1F5H)		\
	__PMC_EV(ARMV8, EVENT_1F6H)		\
	__PMC_EV(ARMV8, EVENT_1F7H)		\
	__PMC_EV(ARMV8, EVENT_1F8H)		\
	__PMC_EV(ARMV8, EVENT_1F9H)		\
	__PMC_EV(ARMV8, EVENT_1FAH)		\
	__PMC_EV(ARMV8, EVENT_1FBH)		\
	__PMC_EV(ARMV8, EVENT_1FCH)		\
	__PMC_EV(ARMV8, EVENT_1FDH)		\
	__PMC_EV(ARMV8, EVENT_1FEH)		\
	__PMC_EV(ARMV8, EVENT_1FFH)		\
	__PMC_EV(ARMV8, EVENT_200H)		\
	__PMC_EV(ARMV8, EVENT_201H)		\
	__PMC_EV(ARMV8, EVENT_202H)		\
	__PMC_EV(ARMV8, EVENT_203H)		\
	__PMC_EV(ARMV8, EVENT_204H)		\
	__PMC_EV(ARMV8, EVENT_205H)		\
	__PMC_EV(ARMV8, EVENT_206H)		\
	__PMC_EV(ARMV8, EVENT_207H)		\
	__PMC_EV(ARMV8, EVENT_208H)		\
	__PMC_EV(ARMV8, EVENT_209H)		\
	__PMC_EV(ARMV8, EVENT_20AH)		\
	__PMC_EV(ARMV8, EVENT_20BH)		\
	__PMC_EV(ARMV8, EVENT_20CH)		\
	__PMC_EV(ARMV8, EVENT_20DH)		\
	__PMC_EV(ARMV8, EVENT_20EH)		\
	__PMC_EV(ARMV8, EVENT_20FH)		\
	__PMC_EV(ARMV8, EVENT_210H)		\
	__PMC_EV(ARMV8, EVENT_211H)		\
	__PMC_EV(ARMV8, EVENT_212H)		\
	__PMC_EV(ARMV8, EVENT_213H)		\
	__PMC_EV(ARMV8, EVENT_214H)		\
	__PMC_EV(ARMV8, EVENT_215H)		\
	__PMC_EV(ARMV8, EVENT_216H)		\
	__PMC_EV(ARMV8, EVENT_217H)		\
	__PMC_EV(ARMV8, EVENT_218H)		\
	__PMC_EV(ARMV8, EVENT_219H)		\
	__PMC_EV(ARMV8, EVENT_21AH)		\
	__PMC_EV(ARMV8, EVENT_21BH)		\
	__PMC_EV(ARMV8, EVENT_21CH)		\
	__PMC_EV(ARMV8, EVENT_21DH)		\
	__PMC_EV(ARMV8, EVENT_21EH)		\
	__PMC_EV(ARMV8, EVENT_21FH)		\
	__PMC_EV(ARMV8, EVENT_220H)		\
	__PMC_EV(ARMV8, EVENT_221H)		\
	__PMC_EV(ARMV8, EVENT_222H)		\
	__PMC_EV(ARMV8, EVENT_223H)		\
	__PMC_EV(ARMV8, EVENT_224H)		\
	__PMC_EV(ARMV8, EVENT_225H)		\
	__PMC_EV(ARMV8, EVENT_226H)		\
	__PMC_EV(ARMV8, EVENT_227H)		\
	__PMC_EV(ARMV8, EVENT_228H)		\
	__PMC_EV(ARMV8, EVENT_229H)		\
	__PMC_EV(ARMV8, EVENT_22AH)		\
	__PMC_EV(ARMV8, EVENT_22BH)		\
	__PMC_EV(ARMV8, EVENT_22CH)		\
	__PMC_EV(ARMV8, EVENT_22DH)		\
	__PMC_EV(ARMV8, EVENT_22EH)		\
	__PMC_EV(ARMV8, EVENT_22FH)		\
	__PMC_EV(ARMV8, EVENT_230H)		\
	__PMC_EV(ARMV8, EVENT_231H)		\
	__PMC_EV(ARMV8, EVENT_232H)		\
	__PMC_EV(ARMV8, EVENT_233H)		\
	__PMC_EV(ARMV8, EVENT_234H)		\
	__PMC_EV(ARMV8, EVENT_235H)		\
	__PMC_EV(ARMV8, EVENT_236H)		\
	__PMC_EV(ARMV8, EVENT_237H)		\
	__PMC_EV(ARMV8, EVENT_238H)		\
	__PMC_EV(ARMV8, EVENT_239H)		\
	__PMC_EV(ARMV8, EVENT_23AH)		\
	__PMC_EV(ARMV8, EVENT_23BH)		\
	__PMC_EV(ARMV8, EVENT_23CH)		\
	__PMC_EV(ARMV8, EVENT_23DH)		\
	__PMC_EV(ARMV8, EVENT_23EH)		\
	__PMC_EV(ARMV8, EVENT_23FH)		\
	__PMC_EV(ARMV8, EVENT_240H)		\
	__PMC_EV(ARMV8, EVENT_241H)		\
	__PMC_EV(ARMV8, EVENT_242H)		\
	__PMC_EV(ARMV8, EVENT_243H)		\
	__PMC_EV(ARMV8, EVENT_244H)		\
	__PMC_EV(ARMV8, EVENT_245H)		\
	__PMC_EV(ARMV8, EVENT_246H)		\
	__PMC_EV(ARMV8, EVENT_247H)		\
	__PMC_EV(ARMV8, EVENT_248H)		\
	__PMC_EV(ARMV8, EVENT_249H)		\
	__PMC_EV(ARMV8, EVENT_24AH)		\
	__PMC_EV(ARMV8, EVENT_24BH)		\
	__PMC_EV(ARMV8, EVENT_24CH)		\
	__PMC_EV(ARMV8, EVENT_24DH)		\
	__PMC_EV(ARMV8, EVENT_24EH)		\
	__PMC_EV(ARMV8, EVENT_24FH)		\
	__PMC_EV(ARMV8, EVENT_250H)		\
	__PMC_EV(ARMV8, EVENT_251H)		\
	__PMC_EV(ARMV8, EVENT_252H)		\
	__PMC_EV(ARMV8, EVENT_253H)		\
	__PMC_EV(ARMV8, EVENT_254H)		\
	__PMC_EV(ARMV8, EVENT_255H)		\
	__PMC_EV(ARMV8, EVENT_256H)		\
	__PMC_EV(ARMV8, EVENT_257H)		\
	__PMC_EV(ARMV8, EVENT_258H)		\
	__PMC_EV(ARMV8, EVENT_259H)		\
	__PMC_EV(ARMV8, EVENT_25AH)		\
	__PMC_EV(ARMV8, EVENT_25BH)		\
	__PMC_EV(ARMV8, EVENT_25CH)		\
	__PMC_EV(ARMV8, EVENT_25DH)		\
	__PMC_EV(ARMV8, EVENT_25EH)		\
	__PMC_EV(ARMV8, EVENT_25FH)		\
	__PMC_EV(ARMV8, EVENT_260H)		\
	__PMC_EV(ARMV8, EVENT_261H)		\
	__PMC_EV(ARMV8, EVENT_262H)		\
	__PMC_EV(ARMV8, EVENT_263H)		\
	__PMC_EV(ARMV8, EVENT_264H)		\
	__PMC_EV(ARMV8, EVENT_265H)		\
	__PMC_EV(ARMV8, EVENT_266H)		\
	__PMC_EV(ARMV8, EVENT_267H)		\
	__PMC_EV(ARMV8, EVENT_268H)		\
	__PMC_EV(ARMV8, EVENT_269H)		\
	__PMC_EV(ARMV8, EVENT_26AH)		\
	__PMC_EV(ARMV8, EVENT_26BH)		\
	__PMC_EV(ARMV8, EVENT_26CH)		\
	__PMC_EV(ARMV8, EVENT_26DH)		\
	__PMC_EV(ARMV8, EVENT_26EH)		\
	__PMC_EV(ARMV8, EVENT_26FH)		\
	__PMC_EV(ARMV8, EVENT_270H)		\
	__PMC_EV(ARMV8, EVENT_271H)		\
	__PMC_EV(ARMV8, EVENT_272H)		\
	__PMC_EV(ARMV8, EVENT_273H)		\
	__PMC_EV(ARMV8, EVENT_274H)		\
	__PMC_EV(ARMV8, EVENT_275H)		\
	__PMC_EV(ARMV8, EVENT_276H)		\
	__PMC_EV(ARMV8, EVENT_277H)		\
	__PMC_EV(ARMV8, EVENT_278H)		\
	__PMC_EV(ARMV8, EVENT_279H)		\
	__PMC_EV(ARMV8, EVENT_27AH)		\
	__PMC_EV(ARMV8, EVENT_27BH)		\
	__PMC_EV(ARMV8, EVENT_27CH)		\
	__PMC_EV(ARMV8, EVENT_27DH)		\
	__PMC_EV(ARMV8, EVENT_27EH)		\
	__PMC_EV(ARMV8, EVENT_27FH)		\
	__PMC_EV(ARMV8, EVENT_280H)		\
	__PMC_EV(ARMV8, EVENT_281H)		\
	__PMC_EV(ARMV8, EVENT_282H)		\
	__PMC_EV(ARMV8, EVENT_283H)		\
	__PMC_EV(ARMV8, EVENT_284H)		\
	__PMC_EV(ARMV8, EVENT_285H)		\
	__PMC_EV(ARMV8, EVENT_286H)		\
	__PMC_EV(ARMV8, EVENT_287H)		\
	__PMC_EV(ARMV8, EVENT_288H)		\
	__PMC_EV(ARMV8, EVENT_289H)		\
	__PMC_EV(ARMV8, EVENT_28AH)		\
	__PMC_EV(ARMV8, EVENT_28BH)		\
	__PMC_EV(ARMV8, EVENT_28CH)		\
	__PMC_EV(ARMV8, EVENT_28DH)		\
	__PMC_EV(ARMV8, EVENT_28EH)		\
	__PMC_EV(ARMV8, EVENT_28FH)		\
	__PMC_EV(ARMV8, EVENT_290H)		\
	__PMC_EV(ARMV8, EVENT_291H)		\
	__PMC_EV(ARMV8, EVENT_292H)		\
	__PMC_EV(ARMV8, EVENT_293H)		\
	__PMC_EV(ARMV8, EVENT_294H)		\
	__PMC_EV(ARMV8, EVENT_295H)		\
	__PMC_EV(ARMV8, EVENT_296H)		\
	__PMC_EV(ARMV8, EVENT_297H)		\
	__PMC_EV(ARMV8, EVENT_298H)		\
	__PMC_EV(ARMV8, EVENT_299H)		\
	__PMC_EV(ARMV8, EVENT_29AH)		\
	__PMC_EV(ARMV8, EVENT_29BH)		\
	__PMC_EV(ARMV8, EVENT_29CH)		\
	__PMC_EV(ARMV8, EVENT_29DH)		\
	__PMC_EV(ARMV8, EVENT_29EH)		\
	__PMC_EV(ARMV8, EVENT_29FH)		\
	__PMC_EV(ARMV8, EVENT_2A0H)		\
	__PMC_EV(ARMV8, EVENT_2A1H)		\
	__PMC_EV(ARMV8, EVENT_2A2H)		\
	__PMC_EV(ARMV8, EVENT_2A3H)		\
	__PMC_EV(ARMV8, EVENT_2A4H)		\
	__PMC_EV(ARMV8, EVENT_2A5H)		\
	__PMC_EV(ARMV8, EVENT_2A6H)		\
	__PMC_EV(ARMV8, EVENT_2A7H)		\
	__PMC_EV(ARMV8, EVENT_2A8H)		\
	__PMC_EV(ARMV8, EVENT_2A9H)		\
	__PMC_EV(ARMV8, EVENT_2AAH)		\
	__PMC_EV(ARMV8, EVENT_2ABH)		\
	__PMC_EV(ARMV8, EVENT_2ACH)		\
	__PMC_EV(ARMV8, EVENT_2ADH)		\
	__PMC_EV(ARMV8, EVENT_2AEH)		\
	__PMC_EV(ARMV8, EVENT_2AFH)		\
	__PMC_EV(ARMV8, EVENT_2B0H)		\
	__PMC_EV(ARMV8, EVENT_2B1H)		\
	__PMC_EV(ARMV8, EVENT_2B2H)		\
	__PMC_EV(ARMV8, EVENT_2B3H)		\
	__PMC_EV(ARMV8, EVENT_2B4H)		\
	__PMC_EV(ARMV8, EVENT_2B5H)		\
	__PMC_EV(ARMV8, EVENT_2B6H)		\
	__PMC_EV(ARMV8, EVENT_2B7H)		\
	__PMC_EV(ARMV8, EVENT_2B8H)		\
	__PMC_EV(ARMV8, EVENT_2B9H)		\
	__PMC_EV(ARMV8, EVENT_2BAH)		\
	__PMC_EV(ARMV8, EVENT_2BBH)		\
	__PMC_EV(ARMV8, EVENT_2BCH)		\
	__PMC_EV(ARMV8, EVENT_2BDH)		\
	__PMC_EV(ARMV8, EVENT_2BEH)		\
	__PMC_EV(ARMV8, EVENT_2BFH)		\
	__PMC_EV(ARMV8, EVENT_2C0H)		\
	__PMC_EV(ARMV8, EVENT_2C1H)		\
	__PMC_EV(ARMV8, EVENT_2C2H)		\
	__PMC_EV(ARMV8, EVENT_2C3H)		\
	__PMC_EV(ARMV8, EVENT_2C4H)		\
	__PMC_EV(ARMV8, EVENT_2C5H)		\
	__PMC_EV(ARMV8, EVENT_2C6H)		\
	__PMC_EV(ARMV8, EVENT_2C7H)		\
	__PMC_EV(ARMV8, EVENT_2C8H)		\
	__PMC_EV(ARMV8, EVENT_2C9H)		\
	__PMC_EV(ARMV8, EVENT_2CAH)		\
	__PMC_EV(ARMV8, EVENT_2CBH)		\
	__PMC_EV(ARMV8, EVENT_2CCH)		\
	__PMC_EV(ARMV8, EVENT_2CDH)		\
	__PMC_EV(ARMV8, EVENT_2CEH)		\
	__PMC_EV(ARMV8, EVENT_2CFH)		\
	__PMC_EV(ARMV8, EVENT_2D0H)		\
	__PMC_EV(ARMV8, EVENT_2D1H)		\
	__PMC_EV(ARMV8, EVENT_2D2H)		\
	__PMC_EV(ARMV8, EVENT_2D3H)		\
	__PMC_EV(ARMV8, EVENT_2D4H)		\
	__PMC_EV(ARMV8, EVENT_2D5H)		\
	__PMC_EV(ARMV8, EVENT_2D6H)		\
	__PMC_EV(ARMV8, EVENT_2D7H)		\
	__PMC_EV(ARMV8, EVENT_2D8H)		\
	__PMC_EV(ARMV8, EVENT_2D9H)		\
	__PMC_EV(ARMV8, EVENT_2DAH)		\
	__PMC_EV(ARMV8, EVENT_2DBH)		\
	__PMC_EV(ARMV8, EVENT_2DCH)		\
	__PMC_EV(ARMV8, EVENT_2DDH)		\
	__PMC_EV(ARMV8, EVENT_2DEH)		\
	__PMC_EV(ARMV8, EVENT_2DFH)		\
	__PMC_EV(ARMV8, EVENT_2E0H)		\
	__PMC_EV(ARMV8, EVENT_2E1H)		\
	__PMC_EV(ARMV8, EVENT_2E2H)		\
	__PMC_EV(ARMV8, EVENT_2E3H)		\
	__PMC_EV(ARMV8, EVENT_2E4H)		\
	__PMC_EV(ARMV8, EVENT_2E5H)		\
	__PMC_EV(ARMV8, EVENT_2E6H)		\
	__PMC_EV(ARMV8, EVENT_2E7H)		\
	__PMC_EV(ARMV8, EVENT_2E8H)		\
	__PMC_EV(ARMV8, EVENT_2E9H)		\
	__PMC_EV(ARMV8, EVENT_2EAH)		\
	__PMC_EV(ARMV8, EVENT_2EBH)		\
	__PMC_EV(ARMV8, EVENT_2ECH)		\
	__PMC_EV(ARMV8, EVENT_2EDH)		\
	__PMC_EV(ARMV8, EVENT_2EEH)		\
	__PMC_EV(ARMV8, EVENT_2EFH)		\
	__PMC_EV(ARMV8, EVENT_2F0H)		\
	__PMC_EV(ARMV8, EVENT_2F1H)		\
	__PMC_EV(ARMV8, EVENT_2F2H)		\
	__PMC_EV(ARMV8, EVENT_2F3H)		\
	__PMC_EV(ARMV8, EVENT_2F4H)		\
	__PMC_EV(ARMV8, EVENT_2F5H)		\
	__PMC_EV(ARMV8, EVENT_2F6H)		\
	__PMC_EV(ARMV8, EVENT_2F7H)		\
	__PMC_EV(ARMV8, EVENT_2F8H)		\
	__PMC_EV(ARMV8, EVENT_2F9H)		\
	__PMC_EV(ARMV8, EVENT_2FAH)		\
	__PMC_EV(ARMV8, EVENT_2FBH)		\
	__PMC_EV(ARMV8, EVENT_2FCH)		\
	__PMC_EV(ARMV8, EVENT_2FDH)		\
	__PMC_EV(ARMV8, EVENT_2FEH)		\
	__PMC_EV(ARMV8, EVENT_2FFH)		\
	__PMC_EV(ARMV8, EVENT_300H)		\
	__PMC_EV(ARMV8, EVENT_301H)		\
	__PMC_EV(ARMV8, EVENT_302H)		\
	__PMC_EV(ARMV8, EVENT_303H)		\
	__PMC_EV(ARMV8, EVENT_304H)		\
	__PMC_EV(ARMV8, EVENT_305H)		\
	__PMC_EV(ARMV8, EVENT_306H)		\
	__PMC_EV(ARMV8, EVENT_307H)		\
	__PMC_EV(ARMV8, EVENT_308H)		\
	__PMC_EV(ARMV8, EVENT_309H)		\
	__PMC_EV(ARMV8, EVENT_30AH)		\
	__PMC_EV(ARMV8, EVENT_30BH)		\
	__PMC_EV(ARMV8, EVENT_30CH)		\
	__PMC_EV(ARMV8, EVENT_30DH)		\
	__PMC_EV(ARMV8, EVENT_30EH)		\
	__PMC_EV(ARMV8, EVENT_30FH)		\
	__PMC_EV(ARMV8, EVENT_310H)		\
	__PMC_EV(ARMV8, EVENT_311H)		\
	__PMC_EV(ARMV8, EVENT_312H)		\
	__PMC_EV(ARMV8, EVENT_313H)		\
	__PMC_EV(ARMV8, EVENT_314H)		\
	__PMC_EV(ARMV8, EVENT_315H)		\
	__PMC_EV(ARMV8, EVENT_316H)		\
	__PMC_EV(ARMV8, EVENT_317H)		\
	__PMC_EV(ARMV8, EVENT_318H)		\
	__PMC_EV(ARMV8, EVENT_319H)		\
	__PMC_EV(ARMV8, EVENT_31AH)		\
	__PMC_EV(ARMV8, EVENT_31BH)		\
	__PMC_EV(ARMV8, EVENT_31CH)		\
	__PMC_EV(ARMV8, EVENT_31DH)		\
	__PMC_EV(ARMV8, EVENT_31EH)		\
	__PMC_EV(ARMV8, EVENT_31FH)		\
	__PMC_EV(ARMV8, EVENT_320H)		\
	__PMC_EV(ARMV8, EVENT_321H)		\
	__PMC_EV(ARMV8, EVENT_322H)		\
	__PMC_EV(ARMV8, EVENT_323H)		\
	__PMC_EV(ARMV8, EVENT_324H)		\
	__PMC_EV(ARMV8, EVENT_325H)		\
	__PMC_EV(ARMV8, EVENT_326H)		\
	__PMC_EV(ARMV8, EVENT_327H)		\
	__PMC_EV(ARMV8, EVENT_328H)		\
	__PMC_EV(ARMV8, EVENT_329H)		\
	__PMC_EV(ARMV8, EVENT_32AH)		\
	__PMC_EV(ARMV8, EVENT_32BH)		\
	__PMC_EV(ARMV8, EVENT_32CH)		\
	__PMC_EV(ARMV8, EVENT_32DH)		\
	__PMC_EV(ARMV8, EVENT_32EH)		\
	__PMC_EV(ARMV8, EVENT_32FH)		\
	__PMC_EV(ARMV8, EVENT_330H)		\
	__PMC_EV(ARMV8, EVENT_331H)		\
	__PMC_EV(ARMV8, EVENT_332H)		\
	__PMC_EV(ARMV8, EVENT_333H)		\
	__PMC_EV(ARMV8, EVENT_334H)		\
	__PMC_EV(ARMV8, EVENT_335H)		\
	__PMC_EV(ARMV8, EVENT_336H)		\
	__PMC_EV(ARMV8, EVENT_337H)		\
	__PMC_EV(ARMV8, EVENT_338H)		\
	__PMC_EV(ARMV8, EVENT_339H)		\
	__PMC_EV(ARMV8, EVENT_33AH)		\
	__PMC_EV(ARMV8, EVENT_33BH)		\
	__PMC_EV(ARMV8, EVENT_33CH)		\
	__PMC_EV(ARMV8, EVENT_33DH)		\
	__PMC_EV(ARMV8, EVENT_33EH)		\
	__PMC_EV(ARMV8, EVENT_33FH)		\
	__PMC_EV(ARMV8, EVENT_340H)		\
	__PMC_EV(ARMV8, EVENT_341H)		\
	__PMC_EV(ARMV8, EVENT_342H)		\
	__PMC_EV(ARMV8, EVENT_343H)		\
	__PMC_EV(ARMV8, EVENT_344H)		\
	__PMC_EV(ARMV8, EVENT_345H)		\
	__PMC_EV(ARMV8, EVENT_346H)		\
	__PMC_EV(ARMV8, EVENT_347H)		\
	__PMC_EV(ARMV8, EVENT_348H)		\
	__PMC_EV(ARMV8, EVENT_349H)		\
	__PMC_EV(ARMV8, EVENT_34AH)		\
	__PMC_EV(ARMV8, EVENT_34BH)		\
	__PMC_EV(ARMV8, EVENT_34CH)		\
	__PMC_EV(ARMV8, EVENT_34DH)		\
	__PMC_EV(ARMV8, EVENT_34EH)		\
	__PMC_EV(ARMV8, EVENT_34FH)		\
	__PMC_EV(ARMV8, EVENT_350H)		\
	__PMC_EV(ARMV8, EVENT_351H)		\
	__PMC_EV(ARMV8, EVENT_352H)		\
	__PMC_EV(ARMV8, EVENT_353H)		\
	__PMC_EV(ARMV8, EVENT_354H)		\
	__PMC_EV(ARMV8, EVENT_355H)		\
	__PMC_EV(ARMV8, EVENT_356H)		\
	__PMC_EV(ARMV8, EVENT_357H)		\
	__PMC_EV(ARMV8, EVENT_358H)		\
	__PMC_EV(ARMV8, EVENT_359H)		\
	__PMC_EV(ARMV8, EVENT_35AH)		\
	__PMC_EV(ARMV8, EVENT_35BH)		\
	__PMC_EV(ARMV8, EVENT_35CH)		\
	__PMC_EV(ARMV8, EVENT_35DH)		\
	__PMC_EV(ARMV8, EVENT_35EH)		\
	__PMC_EV(ARMV8, EVENT_35FH)		\
	__PMC_EV(ARMV8, EVENT_360H)		\
	__PMC_EV(ARMV8, EVENT_361H)		\
	__PMC_EV(ARMV8, EVENT_362H)		\
	__PMC_EV(ARMV8, EVENT_363H)		\
	__PMC_EV(ARMV8, EVENT_364H)		\
	__PMC_EV(ARMV8, EVENT_365H)		\
	__PMC_EV(ARMV8, EVENT_366H)		\
	__PMC_EV(ARMV8, EVENT_367H)		\
	__PMC_EV(ARMV8, EVENT_368H)		\
	__PMC_EV(ARMV8, EVENT_369H)		\
	__PMC_EV(ARMV8, EVENT_36AH)		\
	__PMC_EV(ARMV8, EVENT_36BH)		\
	__PMC_EV(ARMV8, EVENT_36CH)		\
	__PMC_EV(ARMV8, EVENT_36DH)		\
	__PMC_EV(ARMV8, EVENT_36EH)		\
	__PMC_EV(ARMV8, EVENT_36FH)		\
	__PMC_EV(ARMV8, EVENT_370H)		\
	__PMC_EV(ARMV8, EVENT_371H)		\
	__PMC_EV(ARMV8, EVENT_372H)		\
	__PMC_EV(ARMV8, EVENT_373H)		\
	__PMC_EV(ARMV8, EVENT_374H)		\
	__PMC_EV(ARMV8, EVENT_375H)		\
	__PMC_EV(ARMV8, EVENT_376H)		\
	__PMC_EV(ARMV8, EVENT_377H)		\
	__PMC_EV(ARMV8, EVENT_378H)		\
	__PMC_EV(ARMV8, EVENT_379H)		\
	__PMC_EV(ARMV8, EVENT_37AH)		\
	__PMC_EV(ARMV8, EVENT_37BH)		\
	__PMC_EV(ARMV8, EVENT_37CH)		\
	__PMC_EV(ARMV8, EVENT_37DH)		\
	__PMC_EV(ARMV8, EVENT_37EH)		\
	__PMC_EV(ARMV8, EVENT_37FH)		\
	__PMC_EV(ARMV8, EVENT_380H)		\
	__PMC_EV(ARMV8, EVENT_381H)		\
	__PMC_EV(ARMV8, EVENT_382H)		\
	__PMC_EV(ARMV8, EVENT_383H)		\
	__PMC_EV(ARMV8, EVENT_384H)		\
	__PMC_EV(ARMV8, EVENT_385H)		\
	__PMC_EV(ARMV8, EVENT_386H)		\
	__PMC_EV(ARMV8, EVENT_387H)		\
	__PMC_EV(ARMV8, EVENT_388H)		\
	__PMC_EV(ARMV8, EVENT_389H)		\
	__PMC_EV(ARMV8, EVENT_38AH)		\
	__PMC_EV(ARMV8, EVENT_38BH)		\
	__PMC_EV(ARMV8, EVENT_38CH)		\
	__PMC_EV(ARMV8, EVENT_38DH)		\
	__PMC_EV(ARMV8, EVENT_38EH)		\
	__PMC_EV(ARMV8, EVENT_38FH)		\
	__PMC_EV(ARMV8, EVENT_390H)		\
	__PMC_EV(ARMV8, EVENT_391H)		\
	__PMC_EV(ARMV8, EVENT_392H)		\
	__PMC_EV(ARMV8, EVENT_393H)		\
	__PMC_EV(ARMV8, EVENT_394H)		\
	__PMC_EV(ARMV8, EVENT_395H)		\
	__PMC_EV(ARMV8, EVENT_396H)		\
	__PMC_EV(ARMV8, EVENT_397H)		\
	__PMC_EV(ARMV8, EVENT_398H)		\
	__PMC_EV(ARMV8, EVENT_399H)		\
	__PMC_EV(ARMV8, EVENT_39AH)		\
	__PMC_EV(ARMV8, EVENT_39BH)		\
	__PMC_EV(ARMV8, EVENT_39CH)		\
	__PMC_EV(ARMV8, EVENT_39DH)		\
	__PMC_EV(ARMV8, EVENT_39EH)		\
	__PMC_EV(ARMV8, EVENT_39FH)		\
	__PMC_EV(ARMV8, EVENT_3A0H)		\
	__PMC_EV(ARMV8, EVENT_3A1H)		\
	__PMC_EV(ARMV8, EVENT_3A2H)		\
	__PMC_EV(ARMV8, EVENT_3A3H)		\
	__PMC_EV(ARMV8, EVENT_3A4H)		\
	__PMC_EV(ARMV8, EVENT_3A5H)		\
	__PMC_EV(ARMV8, EVENT_3A6H)		\
	__PMC_EV(ARMV8, EVENT_3A7H)		\
	__PMC_EV(ARMV8, EVENT_3A8H)		\
	__PMC_EV(ARMV8, EVENT_3A9H)		\
	__PMC_EV(ARMV8, EVENT_3AAH)		\
	__PMC_EV(ARMV8, EVENT_3ABH)		\
	__PMC_EV(ARMV8, EVENT_3ACH)		\
	__PMC_EV(ARMV8, EVENT_3ADH)		\
	__PMC_EV(ARMV8, EVENT_3AEH)		\
	__PMC_EV(ARMV8, EVENT_3AFH)		\
	__PMC_EV(ARMV8, EVENT_3B0H)		\
	__PMC_EV(ARMV8, EVENT_3B1H)		\
	__PMC_EV(ARMV8, EVENT_3B2H)		\
	__PMC_EV(ARMV8, EVENT_3B3H)		\
	__PMC_EV(ARMV8, EVENT_3B4H)		\
	__PMC_EV(ARMV8, EVENT_3B5H)		\
	__PMC_EV(ARMV8, EVENT_3B6H)		\
	__PMC_EV(ARMV8, EVENT_3B7H)		\
	__PMC_EV(ARMV8, EVENT_3B8H)		\
	__PMC_EV(ARMV8, EVENT_3B9H)		\
	__PMC_EV(ARMV8, EVENT_3BAH)		\
	__PMC_EV(ARMV8, EVENT_3BBH)		\
	__PMC_EV(ARMV8, EVENT_3BCH)		\
	__PMC_EV(ARMV8, EVENT_3BDH)		\
	__PMC_EV(ARMV8, EVENT_3BEH)		\
	__PMC_EV(ARMV8, EVENT_3BFH)		\
	__PMC_EV(ARMV8, EVENT_3C0H)		\
	__PMC_EV(ARMV8, EVENT_3C1H)		\
	__PMC_EV(ARMV8, EVENT_3C2H)		\
	__PMC_EV(ARMV8, EVENT_3C3H)		\
	__PMC_EV(ARMV8, EVENT_3C4H)		\
	__PMC_EV(ARMV8, EVENT_3C5H)		\
	__PMC_EV(ARMV8, EVENT_3C6H)		\
	__PMC_EV(ARMV8, EVENT_3C7H)		\
	__PMC_EV(ARMV8, EVENT_3C8H)		\
	__PMC_EV(ARMV8, EVENT_3C9H)		\
	__PMC_EV(ARMV8, EVENT_3CAH)		\
	__PMC_EV(ARMV8, EVENT_3CBH)		\
	__PMC_EV(ARMV8, EVENT_3CCH)		\
	__PMC_EV(ARMV8, EVENT_3CDH)		\
	__PMC_EV(ARMV8, EVENT_3CEH)		\
	__PMC_EV(ARMV8, EVENT_3CFH)		\
	__PMC_EV(ARMV8, EVENT_3D0H)		\
	__PMC_EV(ARMV8, EVENT_3D1H)		\
	__PMC_EV(ARMV8, EVENT_3D2H)		\
	__PMC_EV(ARMV8, EVENT_3D3H)		\
	__PMC_EV(ARMV8, EVENT_3D4H)		\
	__PMC_EV(ARMV8, EVENT_3D5H)		\
	__PMC_EV(ARMV8, EVENT_3D6H)		\
	__PMC_EV(ARMV8, EVENT_3D7H)		\
	__PMC_EV(ARMV8, EVENT_3D8H)		\
	__PMC_EV(ARMV8, EVENT_3D9H)		\
	__PMC_EV(ARMV8, EVENT_3DAH)		\
	__PMC_EV(ARMV8, EVENT_3DBH)		\
	__PMC_EV(ARMV8, EVENT_3DCH)		\
	__PMC_EV(ARMV8, EVENT_3DDH)		\
	__PMC_EV(ARMV8, EVENT_3DEH)		\
	__PMC_EV(ARMV8, EVENT_3DFH)		\
	__PMC_EV(ARMV8, EVENT_3E0H)		\
	__PMC_EV(ARMV8, EVENT_3E1H)		\
	__PMC_EV(ARMV8, EVENT_3E2H)		\
	__PMC_EV(ARMV8, EVENT_3E3H)		\
	__PMC_EV(ARMV8, EVENT_3E4H)		\
	__PMC_EV(ARMV8, EVENT_3E5H)		\
	__PMC_EV(ARMV8, EVENT_3E6H)		\
	__PMC_EV(ARMV8, EVENT_3E7H)		\
	__PMC_EV(ARMV8, EVENT_3E8H)		\
	__PMC_EV(ARMV8, EVENT_3E9H)		\
	__PMC_EV(ARMV8, EVENT_3EAH)		\
	__PMC_EV(ARMV8, EVENT_3EBH)		\
	__PMC_EV(ARMV8, EVENT_3ECH)		\
	__PMC_EV(ARMV8, EVENT_3EDH)		\
	__PMC_EV(ARMV8, EVENT_3EEH)		\
	__PMC_EV(ARMV8, EVENT_3EFH)		\
	__PMC_EV(ARMV8, EVENT_3F0H)		\
	__PMC_EV(ARMV8, EVENT_3F1H)		\
	__PMC_EV(ARMV8, EVENT_3F2H)		\
	__PMC_EV(ARMV8, EVENT_3F3H)		\
	__PMC_EV(ARMV8, EVENT_3F4H)		\
	__PMC_EV(ARMV8, EVENT_3F5H)		\
	__PMC_EV(ARMV8, EVENT_3F6H)		\
	__PMC_EV(ARMV8, EVENT_3F7H)		\
	__PMC_EV(ARMV8, EVENT_3F8H)		\
	__PMC_EV(ARMV8, EVENT_3F9H)		\
	__PMC_EV(ARMV8, EVENT_3FAH)		\
	__PMC_EV(ARMV8, EVENT_3FBH)		\
	__PMC_EV(ARMV8, EVENT_3FCH)		\
	__PMC_EV(ARMV8, EVENT_3FDH)		\
	__PMC_EV(ARMV8, EVENT_3FEH)		\
	__PMC_EV(ARMV8, EVENT_3FFH)

#define	PMC_EV_ARMV8_FIRST	PMC_EV_ARMV8_EVENT_00H
#define	PMC_EV_ARMV8_LAST	PMC_EV_ARMV8_EVENT_3FFH

#define	__PMC_EV_ALIAS_ARMV8_COMMON()					\
	__PMC_EV_ALIAS("SW_INCR",		ARMV8_EVENT_00H)	\
	__PMC_EV_ALIAS("L1I_CACHE_REFILL",	ARMV8_EVENT_01H)	\
	__PMC_EV_ALIAS("L1I_TLB_REFILL",	ARMV8_EVENT_02H)	\
	__PMC_EV_ALIAS("L1D_CACHE_REFILL",	ARMV8_EVENT_03H)	\
	__PMC_EV_ALIAS("L1D_CACHE",		ARMV8_EVENT_04H)	\
	__PMC_EV_ALIAS("L1D_TLB_REFILL",	ARMV8_EVENT_05H)	\
	__PMC_EV_ALIAS("INST_RETIRED",		ARMV8_EVENT_08H)	\
	__PMC_EV_ALIAS("EXC_TAKEN",		ARMV8_EVENT_09H)	\
	__PMC_EV_ALIAS("EXC_RETURN",		ARMV8_EVENT_0AH)	\
	__PMC_EV_ALIAS("CID_WRITE_RETIRED",	ARMV8_EVENT_0BH)	\
	__PMC_EV_ALIAS("BR_MIS_PRED",		ARMV8_EVENT_10H)	\
	__PMC_EV_ALIAS("CPU_CYCLES",		ARMV8_EVENT_11H)	\
	__PMC_EV_ALIAS("BR_PRED",		ARMV8_EVENT_12H)	\
	__PMC_EV_ALIAS("MEM_ACCESS",		ARMV8_EVENT_13H)	\
	__PMC_EV_ALIAS("L1I_CACHE",		ARMV8_EVENT_14H)	\
	__PMC_EV_ALIAS("L1D_CACHE_WB",		ARMV8_EVENT_15H)	\
	__PMC_EV_ALIAS("L2D_CACHE",		ARMV8_EVENT_16H)	\
	__PMC_EV_ALIAS("L2D_CACHE_REFILL",	ARMV8_EVENT_17H)	\
	__PMC_EV_ALIAS("L2D_CACHE_WB",		ARMV8_EVENT_18H)	\
	__PMC_EV_ALIAS("BUS_ACCESS",		ARMV8_EVENT_19H)	\
	__PMC_EV_ALIAS("MEMORY_ERROR",		ARMV8_EVENT_1AH)	\
	__PMC_EV_ALIAS("BUS_CYCLES",		ARMV8_EVENT_1DH)	\
	__PMC_EV_ALIAS("CHAIN",			ARMV8_EVENT_1EH)	\
	__PMC_EV_ALIAS("BUS_ACCESS_LD",		ARMV8_EVENT_60H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_ST",		ARMV8_EVENT_61H)	\
	__PMC_EV_ALIAS("BR_INDIRECT_SPEC",	ARMV8_EVENT_7AH)	\
	__PMC_EV_ALIAS("EXC_IRQ",		ARMV8_EVENT_86H)	\
	__PMC_EV_ALIAS("EXC_FIQ",		ARMV8_EVENT_87H)

#define	__PMC_EV_ALIAS_ARMV8_CORTEX_A53()				\
	__PMC_EV_ALIAS_ARMV8_COMMON()					\
	__PMC_EV_ALIAS("LD_RETIRED",		ARMV8_EVENT_06H)	\
	__PMC_EV_ALIAS("ST_RETIRED",		ARMV8_EVENT_07H)	\
	__PMC_EV_ALIAS("PC_WRITE_RETIRED",	ARMV8_EVENT_0CH)	\
	__PMC_EV_ALIAS("BR_IMMED_RETIRED",	ARMV8_EVENT_0DH)	\
	__PMC_EV_ALIAS("BR_RETURN_RETIRED",	ARMV8_EVENT_0EH)	\
	__PMC_EV_ALIAS("UNALIGNED_LDST_RETIRED",ARMV8_EVENT_0FH)

#define	__PMC_EV_ALIAS_ARMV8_CORTEX_A57_A76()				\
	__PMC_EV_ALIAS_ARMV8_COMMON()					\
	__PMC_EV_ALIAS("INST_SPEC",		ARMV8_EVENT_1BH)	\
	__PMC_EV_ALIAS("TTBR_WRITE_RETIRED",	ARMV8_EVENT_1CH)	\
	__PMC_EV_ALIAS("L1D_CACHE_LD",		ARMV8_EVENT_40H)	\
	__PMC_EV_ALIAS("L1D_CACHE_ST",		ARMV8_EVENT_41H)	\
	__PMC_EV_ALIAS("L1D_CACHE_REFILL_LD",	ARMV8_EVENT_42H)	\
	__PMC_EV_ALIAS("L1D_CACHE_REFILL_ST",	ARMV8_EVENT_43H)	\
	__PMC_EV_ALIAS("L1D_CACHE_WB_VICTIM",	ARMV8_EVENT_46H)	\
	__PMC_EV_ALIAS("L1D_CACHE_WB_CLEAN",	ARMV8_EVENT_47H)	\
	__PMC_EV_ALIAS("L1D_CACHE_INVAL",	ARMV8_EVENT_48H)	\
	__PMC_EV_ALIAS("L1D_TLB_REFILL_LD",	ARMV8_EVENT_4CH)	\
	__PMC_EV_ALIAS("L1D_TLB_REFILL_ST",	ARMV8_EVENT_4DH)	\
	__PMC_EV_ALIAS("L2D_CACHE_LD",		ARMV8_EVENT_50H)	\
	__PMC_EV_ALIAS("L2D_CACHE_ST",		ARMV8_EVENT_51H)	\
	__PMC_EV_ALIAS("L2D_CACHE_REFILL_LD",	ARMV8_EVENT_52H)	\
	__PMC_EV_ALIAS("L2D_CACHE_REFILL_ST",	ARMV8_EVENT_53H)	\
	__PMC_EV_ALIAS("L2D_CACHE_WB_VICTIM",	ARMV8_EVENT_56H)	\
	__PMC_EV_ALIAS("L2D_CACHE_WB_CLEAN",	ARMV8_EVENT_57H)	\
	__PMC_EV_ALIAS("L2D_CACHE_INVAL",	ARMV8_EVENT_58H)	\
	__PMC_EV_ALIAS("MEM_ACCESS_LD",		ARMV8_EVENT_66H)	\
	__PMC_EV_ALIAS("MEM_ACCESS_ST",		ARMV8_EVENT_67H)	\
	__PMC_EV_ALIAS("UNALIGNED_LD_SPEC",	ARMV8_EVENT_68H)	\
	__PMC_EV_ALIAS("UNALIGNED_ST_SPEC",	ARMV8_EVENT_69H)	\
	__PMC_EV_ALIAS("UNALIGNED_LDST_SPEC",	ARMV8_EVENT_6AH)	\
	__PMC_EV_ALIAS("LDREX_SPEC",		ARMV8_EVENT_6CH)	\
	__PMC_EV_ALIAS("STREX_PASS_SPEC",	ARMV8_EVENT_6DH)	\
	__PMC_EV_ALIAS("STREX_FAIL_SPEC",	ARMV8_EVENT_6EH)	\
	__PMC_EV_ALIAS("LD_SPEC",		ARMV8_EVENT_70H)	\
	__PMC_EV_ALIAS("ST_SPEC",		ARMV8_EVENT_71H)	\
	__PMC_EV_ALIAS("LDST_SPEC",		ARMV8_EVENT_72H)	\
	__PMC_EV_ALIAS("DP_SPEC",		ARMV8_EVENT_73H)	\
	__PMC_EV_ALIAS("ASE_SPEC",		ARMV8_EVENT_74H)	\
	__PMC_EV_ALIAS("VFP_SPEC",		ARMV8_EVENT_75H)	\
	__PMC_EV_ALIAS("PC_WRITE_SPEC",		ARMV8_EVENT_76H)	\
	__PMC_EV_ALIAS("CRYPTO_SPEC",		ARMV8_EVENT_77H)	\
	__PMC_EV_ALIAS("BR_IMMED_SPEC",		ARMV8_EVENT_78H)	\
	__PMC_EV_ALIAS("BR_RETURN_SPEC",	ARMV8_EVENT_79H)	\
	__PMC_EV_ALIAS("ISB_SPEC",		ARMV8_EVENT_7CH)	\
	__PMC_EV_ALIAS("DSB_SPEC",		ARMV8_EVENT_7DH)	\
	__PMC_EV_ALIAS("DMB_SPEC",		ARMV8_EVENT_7EH)	\
	__PMC_EV_ALIAS("EXC_UNDEF",		ARMV8_EVENT_81H)	\
	__PMC_EV_ALIAS("EXC_SVC",		ARMV8_EVENT_82H)	\
	__PMC_EV_ALIAS("EXC_PABORT",		ARMV8_EVENT_83H)	\
	__PMC_EV_ALIAS("EXC_DABORT",		ARMV8_EVENT_84H)	\
	__PMC_EV_ALIAS("EXC_SMC",		ARMV8_EVENT_88H)	\
	__PMC_EV_ALIAS("EXC_HVC",		ARMV8_EVENT_8AH)	\
	__PMC_EV_ALIAS("EXC_TRAP_PABORT",	ARMV8_EVENT_8BH)	\
	__PMC_EV_ALIAS("EXC_TRAP_DABORT",	ARMV8_EVENT_8CH)	\
	__PMC_EV_ALIAS("EXC_TRAP_OTHER",	ARMV8_EVENT_8DH)	\
	__PMC_EV_ALIAS("EXC_TRAP_IRQ",		ARMV8_EVENT_8EH)	\
	__PMC_EV_ALIAS("EXC_TRAP_FIQ",		ARMV8_EVENT_8FH)	\
	__PMC_EV_ALIAS("RC_LD_SPEC",		ARMV8_EVENT_90H)	\
	__PMC_EV_ALIAS("RC_ST_SPEC",		ARMV8_EVENT_91H)

#define	__PMC_EV_ALIAS_ARMV8_CORTEX_A57()				\
	__PMC_EV_ALIAS_ARMV8_CORTEX_A57_A76()				\
	__PMC_EV_ALIAS("BUS_ACCESS_SHARED",	ARMV8_EVENT_62H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_NOT_SHARED",	ARMV8_EVENT_63H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_NORMAL",	ARMV8_EVENT_64H)	\
	__PMC_EV_ALIAS("BUS_ACCESS_PERIPH",	ARMV8_EVENT_65H)

#define	__PMC_EV_ALIAS_ARMV8_CORTEX_A76()				\
	__PMC_EV_ALIAS_ARMV8_CORTEX_A57_A76()				\
	__PMC_EV_ALIAS("L2D_CACHE_ALLOCATE",	ARMV8_EVENT_20H)	\
	__PMC_EV_ALIAS("BR_RETIRED",		ARMV8_EVENT_21H)	\
	__PMC_EV_ALIAS("BR_MIS_PRED_RETIRED",	ARMV8_EVENT_22H)	\
	__PMC_EV_ALIAS("STALL_FRONTEND",	ARMV8_EVENT_23H)	\
	__PMC_EV_ALIAS("STALL_BACKEND",		ARMV8_EVENT_24H)	\
	__PMC_EV_ALIAS("L1D_TLB",		ARMV8_EVENT_25H)	\
	__PMC_EV_ALIAS("L1I_TLB",		ARMV8_EVENT_26H)	\
	__PMC_EV_ALIAS("L3D_CACHE_ALLOCATE",	ARMV8_EVENT_29H)	\
	__PMC_EV_ALIAS("L3D_CACHE_REFILL",	ARMV8_EVENT_2AH)	\
	__PMC_EV_ALIAS("L3D_CACHE",		ARMV8_EVENT_2BH)	\
	__PMC_EV_ALIAS("L2D_TLB_REFILL",	ARMV8_EVENT_2DH)	\
	__PMC_EV_ALIAS("L2D_TLB",		ARMV8_EVENT_2FH)	\
	__PMC_EV_ALIAS("REMOTE_ACCESS",		ARMV8_EVENT_31H)	\
	__PMC_EV_ALIAS("DTLB_WALK",		ARMV8_EVENT_34H)	\
	__PMC_EV_ALIAS("ITLB_WALK",		ARMV8_EVENT_35H)	\
	__PMC_EV_ALIAS("LL_CACHE_RD",		ARMV8_EVENT_36H)	\
	__PMC_EV_ALIAS("LL_CACHE_MISS_RD",	ARMV8_EVENT_37H)	\
	__PMC_EV_ALIAS("L1D_CACHE_REFILL_INNER", ARMV8_EVENT_44H)	\
	__PMC_EV_ALIAS("L1D_CACHE_REFILL_OUTER", ARMV8_EVENT_45H)	\
	__PMC_EV_ALIAS("L1D_TLB_RD",		ARMV8_EVENT_4EH)	\
	__PMC_EV_ALIAS("L1D_TLB_WR",		ARMV8_EVENT_4FH)	\
	__PMC_EV_ALIAS("L2D_TLB_REFILL_RD",	ARMV8_EVENT_5CH)	\
	__PMC_EV_ALIAS("L2D_TLB_REFILL_WR",	ARMV8_EVENT_5DH)	\
	__PMC_EV_ALIAS("L2D_TLB_RD",		ARMV8_EVENT_5EH)	\
	__PMC_EV_ALIAS("L2D_TLB_WR",		ARMV8_EVENT_5FH)	\
	__PMC_EV_ALIAS("STREX_SPEC",		ARMV8_EVENT_6FH)	\
	__PMC_EV_ALIAS("L3_CACHE_RD",		ARMV8_EVENT_A0H)

/*
 * ARM DMC-620 memory controller counters.
 */

#define	__PMC_EV_DMC620_PMU_CD2()					\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_cycle_count)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_allocate)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_queue_depth)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_waiting_for_wr_data)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_read_backlog)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_waiting_for_mi)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_hazard_resolution)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_enqueue)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_arbitrate)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_lrank_turnaround_activate)	\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_prank_turnaround_activate)	\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_read_depth)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_write_depth)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_highhigh_qos_depth)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_high_qos_depth)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_medium_qos_depth)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_low_qos_depth)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_activate)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_rdwr)				\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_refresh)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_training_request)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_t_mac_tracker)			\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_bk_fsm_tracker)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_bk_open_tracker)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_ranks_in_pwr_down)		\
	__PMC_EV(DMC620_PMU_CD2, clkdiv2_ranks_in_sref)			\

#define	__PMC_EV_DMC620_PMU_C()						\
	__PMC_EV(DMC620_PMU_C, clk_cycle_count)				\
	__PMC_EV(DMC620_PMU_C, clk_request)				\
	__PMC_EV(DMC620_PMU_C, clk_upload_stall)

#define	PMC_EV_DMC620_PMU_CD2_FIRST	PMC_EV_DMC620_PMU_CD2_clkdiv2_cycle_count
#define	PMC_EV_DMC620_PMU_CD2_LAST	PMC_EV_DMC620_PMU_CD2_clkdiv2_ranks_in_sref
#define	PMC_EV_DMC620_PMU_C_FIRST	PMC_EV_DMC620_PMU_C_clk_cycle_count
#define	PMC_EV_DMC620_PMU_C_LAST	PMC_EV_DMC620_PMU_C_clk_upload_stall

/*
 * Arm CMN-600 Coherent Mesh Network controller counters.
 */

#define	__PMC_EV_CMN600_PMU()					\
	__PMC_EV(CMN600_PMU, dn_rxreq_dvmop)			\
	__PMC_EV(CMN600_PMU, dn_rxreq_dvmsync)			\
	__PMC_EV(CMN600_PMU, dn_rxreq_dvmop_vmid_filtered)	\
	__PMC_EV(CMN600_PMU, dn_rxreq_retried)			\
	__PMC_EV(CMN600_PMU, dn_rxreq_trk_occupancy)		\
	__PMC_EV(CMN600_PMU, dn_rxreq_tlbi_dvmop)		\
	__PMC_EV(CMN600_PMU, dn_rxreq_bpi_dvmop)		\
	__PMC_EV(CMN600_PMU, dn_rxreq_pici_dvmop)		\
	__PMC_EV(CMN600_PMU, dn_rxreq_vivi_dvmop)		\
	__PMC_EV(CMN600_PMU, dn_rxreq_dvmop_other_filtered)	\
	__PMC_EV(CMN600_PMU, dn_rxreq_snp_sent)			\
	__PMC_EV(CMN600_PMU, dn_rxreq_snp_stalled)		\
	__PMC_EV(CMN600_PMU, dn_rxreq_trk_full)			\
	__PMC_EV(CMN600_PMU, hnf_cache_miss)			\
	__PMC_EV(CMN600_PMU, hnf_slc_sf_cache_access)		\
	__PMC_EV(CMN600_PMU, hnf_cache_fill)			\
	__PMC_EV(CMN600_PMU, hnf_pocq_retry)			\
	__PMC_EV(CMN600_PMU, hnf_pocq_reqs_recvd)		\
	__PMC_EV(CMN600_PMU, hnf_sf_hit)			\
	__PMC_EV(CMN600_PMU, hnf_sf_evictions)			\
	__PMC_EV(CMN600_PMU, hnf_dir_snoops_sent)		\
	__PMC_EV(CMN600_PMU, hnf_brd_snoops_sent)		\
	__PMC_EV(CMN600_PMU, hnf_slc_eviction)			\
	__PMC_EV(CMN600_PMU, hnf_slc_fill_invalid_way)		\
	__PMC_EV(CMN600_PMU, hnf_mc_retries)			\
	__PMC_EV(CMN600_PMU, hnf_mc_reqs)			\
	__PMC_EV(CMN600_PMU, hnf_qos_hh_retry)			\
	__PMC_EV(CMN600_PMU, hnf_qos_pocq)			\
	__PMC_EV(CMN600_PMU, hnf_pocq_addrhaz)			\
	__PMC_EV(CMN600_PMU, hnf_pocq_atomic_addrhaz)		\
	__PMC_EV(CMN600_PMU, hnf_ld_st_swp_adq_full)		\
	__PMC_EV(CMN600_PMU, hnf_cmp_adq_full)			\
	__PMC_EV(CMN600_PMU, hnf_txdat_stall)			\
	__PMC_EV(CMN600_PMU, hnf_txrsp_stall)			\
	__PMC_EV(CMN600_PMU, hnf_seq_full)			\
	__PMC_EV(CMN600_PMU, hnf_seq_hit)			\
	__PMC_EV(CMN600_PMU, hnf_snp_sent)			\
	__PMC_EV(CMN600_PMU, hnf_sfbi_dir_snp_sent)		\
	__PMC_EV(CMN600_PMU, hnf_sfbi_brd_snp_sent)		\
	__PMC_EV(CMN600_PMU, hnf_snp_sent_untrk)		\
	__PMC_EV(CMN600_PMU, hnf_intv_dirty)			\
	__PMC_EV(CMN600_PMU, hnf_stash_snp_sent)		\
	__PMC_EV(CMN600_PMU, hnf_stash_data_pull)		\
	__PMC_EV(CMN600_PMU, hnf_snp_fwded)			\
	__PMC_EV(CMN600_PMU, hni_rrt_rd_occ_cnt_ovfl)		\
	__PMC_EV(CMN600_PMU, hni_rrt_wr_occ_cnt_ovfl)		\
	__PMC_EV(CMN600_PMU, hni_rdt_rd_occ_cnt_ovfl)		\
	__PMC_EV(CMN600_PMU, hni_rdt_wr_occ_cnt_ovfl)		\
	__PMC_EV(CMN600_PMU, hni_wdb_occ_cnt_ovfl)		\
	__PMC_EV(CMN600_PMU, hni_rrt_rd_alloc)			\
	__PMC_EV(CMN600_PMU, hni_rrt_wr_alloc)			\
	__PMC_EV(CMN600_PMU, hni_rdt_rd_alloc)			\
	__PMC_EV(CMN600_PMU, hni_rdt_wr_alloc)			\
	__PMC_EV(CMN600_PMU, hni_wdb_alloc)			\
	__PMC_EV(CMN600_PMU, hni_txrsp_retryack)		\
	__PMC_EV(CMN600_PMU, hni_arvalid_no_arready)		\
	__PMC_EV(CMN600_PMU, hni_arready_no_arvalid)		\
	__PMC_EV(CMN600_PMU, hni_awvalid_no_awready)		\
	__PMC_EV(CMN600_PMU, hni_awready_no_awvalid)		\
	__PMC_EV(CMN600_PMU, hni_wvalid_no_wready)		\
	__PMC_EV(CMN600_PMU, hni_txdat_stall)			\
	__PMC_EV(CMN600_PMU, hni_nonpcie_serialization)		\
	__PMC_EV(CMN600_PMU, hni_pcie_serialization)		\
	__PMC_EV(CMN600_PMU, xp_txflit_valid)			\
	__PMC_EV(CMN600_PMU, xp_txflit_stall)			\
	__PMC_EV(CMN600_PMU, xp_partial_dat_flit)		\
	__PMC_EV(CMN600_PMU, sbsx_rd_req)			\
	__PMC_EV(CMN600_PMU, sbsx_wr_req)			\
	__PMC_EV(CMN600_PMU, sbsx_cmo_req)			\
	__PMC_EV(CMN600_PMU, sbsx_txrsp_retryack)		\
	__PMC_EV(CMN600_PMU, sbsx_txdat_flitv)			\
	__PMC_EV(CMN600_PMU, sbsx_txrsp_flitv)			\
	__PMC_EV(CMN600_PMU, sbsx_rd_req_trkr_occ_cnt_ovfl)	\
	__PMC_EV(CMN600_PMU, sbsx_wr_req_trkr_occ_cnt_ovfl)	\
	__PMC_EV(CMN600_PMU, sbsx_cmo_req_trkr_occ_cnt_ovfl)	\
	__PMC_EV(CMN600_PMU, sbsx_wdb_occ_cnt_ovfl)		\
	__PMC_EV(CMN600_PMU, sbsx_rd_axi_trkr_occ_cnt_ovfl)	\
	__PMC_EV(CMN600_PMU, sbsx_cmo_axi_trkr_occ_cnt_ovfl)	\
	__PMC_EV(CMN600_PMU, sbsx_arvalid_no_arready)		\
	__PMC_EV(CMN600_PMU, sbsx_awvalid_no_awready)		\
	__PMC_EV(CMN600_PMU, sbsx_wvalid_no_wready)		\
	__PMC_EV(CMN600_PMU, sbsx_txdat_stall)			\
	__PMC_EV(CMN600_PMU, sbsx_txrsp_stall)			\
	__PMC_EV(CMN600_PMU, rnd_s0_rdata_beats)		\
	__PMC_EV(CMN600_PMU, rnd_s1_rdata_beats)		\
	__PMC_EV(CMN600_PMU, rnd_s2_rdata_beats)		\
	__PMC_EV(CMN600_PMU, rnd_rxdat_flits)			\
	__PMC_EV(CMN600_PMU, rnd_txdat_flits)			\
	__PMC_EV(CMN600_PMU, rnd_txreq_flits_total)		\
	__PMC_EV(CMN600_PMU, rnd_txreq_flits_retried)		\
	__PMC_EV(CMN600_PMU, rnd_rrt_occ_ovfl)			\
	__PMC_EV(CMN600_PMU, rnd_wrt_occ_ovfl)			\
	__PMC_EV(CMN600_PMU, rnd_txreq_flits_replayed)		\
	__PMC_EV(CMN600_PMU, rnd_wrcancel_sent)			\
	__PMC_EV(CMN600_PMU, rnd_s0_wdata_beats)		\
	__PMC_EV(CMN600_PMU, rnd_s1_wdata_beats)		\
	__PMC_EV(CMN600_PMU, rnd_s2_wdata_beats)		\
	__PMC_EV(CMN600_PMU, rnd_rrt_alloc)			\
	__PMC_EV(CMN600_PMU, rnd_wrt_alloc)			\
	__PMC_EV(CMN600_PMU, rnd_rdb_unord)			\
	__PMC_EV(CMN600_PMU, rnd_rdb_replay)			\
	__PMC_EV(CMN600_PMU, rnd_rdb_hybrid)			\
	__PMC_EV(CMN600_PMU, rnd_rdb_ord)			\
	__PMC_EV(CMN600_PMU, rni_s0_rdata_beats)		\
	__PMC_EV(CMN600_PMU, rni_s1_rdata_beats)		\
	__PMC_EV(CMN600_PMU, rni_s2_rdata_beats)		\
	__PMC_EV(CMN600_PMU, rni_rxdat_flits)			\
	__PMC_EV(CMN600_PMU, rni_txdat_flits)			\
	__PMC_EV(CMN600_PMU, rni_txreq_flits_total)		\
	__PMC_EV(CMN600_PMU, rni_txreq_flits_retried)		\
	__PMC_EV(CMN600_PMU, rni_rrt_occ_ovfl)			\
	__PMC_EV(CMN600_PMU, rni_wrt_occ_ovfl)			\
	__PMC_EV(CMN600_PMU, rni_txreq_flits_replayed)		\
	__PMC_EV(CMN600_PMU, rni_wrcancel_sent)			\
	__PMC_EV(CMN600_PMU, rni_s0_wdata_beats)		\
	__PMC_EV(CMN600_PMU, rni_s1_wdata_beats)		\
	__PMC_EV(CMN600_PMU, rni_s2_wdata_beats)		\
	__PMC_EV(CMN600_PMU, rni_rrt_alloc)			\
	__PMC_EV(CMN600_PMU, rni_wrt_alloc)			\
	__PMC_EV(CMN600_PMU, rni_rdb_unord)			\
	__PMC_EV(CMN600_PMU, rni_rdb_replay)			\
	__PMC_EV(CMN600_PMU, rni_rdb_hybrid)			\
	__PMC_EV(CMN600_PMU, rni_rdb_ord)			\
	__PMC_EV(CMN600_PMU, cxha_rddatbyp)			\
	__PMC_EV(CMN600_PMU, cxha_chirsp_up_stall)		\
	__PMC_EV(CMN600_PMU, cxha_chidat_up_stall)		\
	__PMC_EV(CMN600_PMU, cxha_snppcrd_lnk0_stall)		\
	__PMC_EV(CMN600_PMU, cxha_snppcrd_lnk1_stall)		\
	__PMC_EV(CMN600_PMU, cxha_snppcrd_lnk2_stall)		\
	__PMC_EV(CMN600_PMU, cxha_reqtrk_occ)			\
	__PMC_EV(CMN600_PMU, cxha_rdb_occ)			\
	__PMC_EV(CMN600_PMU, cxha_rdbbyp_occ)			\
	__PMC_EV(CMN600_PMU, cxha_wdb_occ)			\
	__PMC_EV(CMN600_PMU, cxha_snptrk_occ)			\
	__PMC_EV(CMN600_PMU, cxha_sdb_occ)			\
	__PMC_EV(CMN600_PMU, cxha_snphaz_occ)			\
	__PMC_EV(CMN600_PMU, cxra_req_trk_occ)			\
	__PMC_EV(CMN600_PMU, cxra_snp_trk_occ)			\
	__PMC_EV(CMN600_PMU, cxra_rd_dat_buf_occ)		\
	__PMC_EV(CMN600_PMU, cxra_wr_dat_buf_occ)		\
	__PMC_EV(CMN600_PMU, cxra_snp_sink_buf_occ)		\
	__PMC_EV(CMN600_PMU, cxra_snp_bcasts)			\
	__PMC_EV(CMN600_PMU, cxra_req_chains)			\
	__PMC_EV(CMN600_PMU, cxra_req_chain_avg_len)		\
	__PMC_EV(CMN600_PMU, cxra_chi_rsp_upload_stalls)	\
	__PMC_EV(CMN600_PMU, cxra_chi_dat_upload_stalls)	\
	__PMC_EV(CMN600_PMU, cxra_dat_pcrd_stalls_lnk0)		\
	__PMC_EV(CMN600_PMU, cxra_dat_pcrd_stalls_lnk1)		\
	__PMC_EV(CMN600_PMU, cxra_dat_pcrd_stalls_lnk2)		\
	__PMC_EV(CMN600_PMU, cxra_req_pcrd_stalls_lnk0)		\
	__PMC_EV(CMN600_PMU, cxra_req_pcrd_stalls_lnk1)		\
	__PMC_EV(CMN600_PMU, cxra_req_pcrd_stalls_lnk2)		\
	__PMC_EV(CMN600_PMU, cxra_ext_rsp_stall)		\
	__PMC_EV(CMN600_PMU, cxra_ext_dat_stall)		\
	__PMC_EV(CMN600_PMU, cxla_rx_tlp_link0)			\
	__PMC_EV(CMN600_PMU, cxla_rx_tlp_link1)			\
	__PMC_EV(CMN600_PMU, cxla_rx_tlp_link2)			\
	__PMC_EV(CMN600_PMU, cxla_tx_tlp_link0)			\
	__PMC_EV(CMN600_PMU, cxla_tx_tlp_link1)			\
	__PMC_EV(CMN600_PMU, cxla_tx_tlp_link2)			\
	__PMC_EV(CMN600_PMU, cxla_rx_cxs_link0)			\
	__PMC_EV(CMN600_PMU, cxla_rx_cxs_link1)			\
	__PMC_EV(CMN600_PMU, cxla_rx_cxs_link2)			\
	__PMC_EV(CMN600_PMU, cxla_tx_cxs_link0)			\
	__PMC_EV(CMN600_PMU, cxla_tx_cxs_link1)			\
	__PMC_EV(CMN600_PMU, cxla_tx_cxs_link2)			\
	__PMC_EV(CMN600_PMU, cxla_avg_rx_tlp_sz_dws)		\
	__PMC_EV(CMN600_PMU, cxla_avg_tx_tlp_sz_dws)		\
	__PMC_EV(CMN600_PMU, cxla_avg_rx_tlp_sz_ccix_msg)	\
	__PMC_EV(CMN600_PMU, cxla_avg_tx_tlp_sz_ccix_msg)	\
	__PMC_EV(CMN600_PMU, cxla_avg_sz_rx_cxs_dw_beat)	\
	__PMC_EV(CMN600_PMU, cxla_avg_sz_tx_cxs_dw_beat)	\
	__PMC_EV(CMN600_PMU, cxla_tx_cxs_link_credit_backpressure) \
	__PMC_EV(CMN600_PMU, cxla_rx_tlp_buffer_full)		\
	__PMC_EV(CMN600_PMU, cxla_tx_tlp_buffer_full)		\
	__PMC_EV(CMN600_PMU, cxla_avg_latency_process_rx_tlp)	\
	__PMC_EV(CMN600_PMU, cxla_avg_latency_form_tx_tlp)


#define	PMC_EV_CMN600_PMU_FIRST	PMC_EV_CMN600_PMU_dn_rxreq_dvmop
#define	PMC_EV_CMN600_PMU_LAST	\
		PMC_EV_CMN600_PMU_cxla_avg_latency_form_tx_tlp

#define __PMC_EV_PPC7450()						\
	__PMC_EV(PPC7450, CYCLE)					\
	__PMC_EV(PPC7450, INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, TLB_BIT_TRANSITIONS)				\
	__PMC_EV(PPC7450, INSTR_DISPATCHED)				\
	__PMC_EV(PPC7450, PMON_EXCEPT)					\
	__PMC_EV(PPC7450, PMON_SIG)					\
	__PMC_EV(PPC7450, VPU_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, VFPU_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, VIU1_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, VIU2_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, MTVSCR_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, MTVRSAVE_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, VPU_INSTR_WAIT_CYCLES)			\
	__PMC_EV(PPC7450, VFPU_INSTR_WAIT_CYCLES)			\
	__PMC_EV(PPC7450, VIU1_INSTR_WAIT_CYCLES)			\
	__PMC_EV(PPC7450, VIU2_INSTR_WAIT_CYCLES)			\
	__PMC_EV(PPC7450, MFVSCR_SYNC_CYCLES)				\
	__PMC_EV(PPC7450, VSCR_SAT_SET)					\
	__PMC_EV(PPC7450, STORE_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, L1_INSTR_CACHE_MISSES)			\
	__PMC_EV(PPC7450, L1_DATA_SNOOPS)				\
	__PMC_EV(PPC7450, UNRESOLVED_BRANCHES)				\
	__PMC_EV(PPC7450, SPEC_BUFFER_CYCLES)				\
	__PMC_EV(PPC7450, BRANCH_UNIT_STALL_CYCLES)			\
	__PMC_EV(PPC7450, TRUE_BRANCH_TARGET_HITS)			\
	__PMC_EV(PPC7450, BRANCH_LINK_STAC_PREDICTED)			\
	__PMC_EV(PPC7450, GPR_ISSUE_QUEUE_DISPATCHES)			\
	__PMC_EV(PPC7450, CYCLES_THREE_INSTR_DISPATCHED)		\
	__PMC_EV(PPC7450, THRESHOLD_INSTR_QUEUE_ENTRIES_CYCLES)		\
	__PMC_EV(PPC7450, THRESHOLD_VEC_INSTR_QUEUE_ENTRIES_CYCLES)	\
	__PMC_EV(PPC7450, CYCLES_NO_COMPLETED_INSTRS)			\
	__PMC_EV(PPC7450, IU2_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, BRANCHES_COMPLETED)				\
	__PMC_EV(PPC7450, EIEIO_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, MTSPR_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, SC_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, LS_LM_COMPLETED)				\
	__PMC_EV(PPC7450, ITLB_HW_TABLE_SEARCH_CYCLES)			\
	__PMC_EV(PPC7450, DTLB_HW_SEARCH_CYCLES_OVER_THRESHOLD)		\
	__PMC_EV(PPC7450, L1_INSTR_CACHE_ACCESSES)			\
	__PMC_EV(PPC7450, INSTR_BKPT_MATCHES)				\
	__PMC_EV(PPC7450, L1_DATA_CACHE_LOAD_MISS_CYCLES_OVER_THRESHOLD)\
	__PMC_EV(PPC7450, L1_DATA_SNOOP_HIT_ON_MODIFIED)		\
	__PMC_EV(PPC7450, LOAD_MISS_ALIAS)				\
	__PMC_EV(PPC7450, LOAD_MISS_ALIAS_ON_TOUCH)			\
	__PMC_EV(PPC7450, TOUCH_ALIAS)					\
	__PMC_EV(PPC7450, L1_DATA_SNOOP_HIT_CASTOUT_QUEUE)		\
	__PMC_EV(PPC7450, L1_DATA_SNOOP_HIT_CASTOUT)			\
	__PMC_EV(PPC7450, L1_DATA_SNOOP_HITS)				\
	__PMC_EV(PPC7450, WRITE_THROUGH_STORES)				\
	__PMC_EV(PPC7450, CACHE_INHIBITED_STORES)			\
	__PMC_EV(PPC7450, L1_DATA_LOAD_HIT)				\
	__PMC_EV(PPC7450, L1_DATA_TOUCH_HIT)				\
	__PMC_EV(PPC7450, L1_DATA_STORE_HIT)				\
	__PMC_EV(PPC7450, L1_DATA_TOTAL_HITS)				\
	__PMC_EV(PPC7450, DST_INSTR_DISPATCHED)				\
	__PMC_EV(PPC7450, REFRESHED_DSTS)				\
	__PMC_EV(PPC7450, SUCCESSFUL_DST_TABLE_SEARCHES)		\
	__PMC_EV(PPC7450, DSS_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, DST_STREAM_0_CACHE_LINE_FETCHES)		\
	__PMC_EV(PPC7450, VTQ_SUSPENDS_DUE_TO_CTX_CHANGE)		\
	__PMC_EV(PPC7450, VTQ_LINE_FETCH_HIT)				\
	__PMC_EV(PPC7450, VEC_LOAD_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, FP_STORE_INSTR_COMPLETED_IN_LSU)		\
	__PMC_EV(PPC7450, FPU_RENORMALIZATION)				\
	__PMC_EV(PPC7450, FPU_DENORMALIZATION)				\
	__PMC_EV(PPC7450, FP_STORE_CAUSES_STALL_IN_LSU)			\
	__PMC_EV(PPC7450, LD_ST_TRUE_ALIAS_STALL)			\
	__PMC_EV(PPC7450, LSU_INDEXED_ALIAS_STALL)			\
	__PMC_EV(PPC7450, LSU_ALIAS_VS_FSQ_WB0_WB1)			\
	__PMC_EV(PPC7450, LSU_ALIAS_VS_CSQ)				\
	__PMC_EV(PPC7450, LSU_LOAD_HIT_LINE_ALIAS_VS_CSQ0)		\
	__PMC_EV(PPC7450, LSU_LOAD_MISS_LINE_ALIAS_VS_CSQ0)		\
	__PMC_EV(PPC7450, LSU_TOUCH_LINE_ALIAS_VS_FSQ_WB0_WB1)		\
	__PMC_EV(PPC7450, LSU_TOUCH_ALIAS_VS_CSQ)			\
	__PMC_EV(PPC7450, LSU_LMQ_FULL_STALL)				\
	__PMC_EV(PPC7450, FP_LOAD_INSTR_COMPLETED_IN_LSU)		\
	__PMC_EV(PPC7450, FP_LOAD_SINGLE_INSTR_COMPLETED_IN_LSU)	\
	__PMC_EV(PPC7450, FP_LOAD_DOUBLE_COMPLETED_IN_LSU)		\
	__PMC_EV(PPC7450, LSU_RA_LATCH_STALL)				\
	__PMC_EV(PPC7450, LSU_LOAD_VS_STORE_QUEUE_ALIAS_STALL)		\
	__PMC_EV(PPC7450, LSU_LMQ_INDEX_ALIAS)				\
	__PMC_EV(PPC7450, LSU_STORE_QUEUE_INDEX_ALIAS)			\
	__PMC_EV(PPC7450, LSU_CSQ_FORWARDING)				\
	__PMC_EV(PPC7450, LSU_MISALIGNED_LOAD_FINISH)			\
	__PMC_EV(PPC7450, LSU_MISALIGN_STORE_COMPLETED)			\
	__PMC_EV(PPC7450, LSU_MISALIGN_STALL)				\
	__PMC_EV(PPC7450, FP_ONE_QUARTER_FPSCR_RENAMES_BUSY)		\
	__PMC_EV(PPC7450, FP_ONE_HALF_FPSCR_RENAMES_BUSY)		\
	__PMC_EV(PPC7450, FP_THREE_QUARTERS_FPSCR_RENAMES_BUSY)		\
	__PMC_EV(PPC7450, FP_ALL_FPSCR_RENAMES_BUSY)			\
	__PMC_EV(PPC7450, FP_DENORMALIZED_RESULT)			\
	__PMC_EV(PPC7450, L1_DATA_TOTAL_MISSES)				\
	__PMC_EV(PPC7450, DISPATCHES_TO_FPR_ISSUE_QUEUE)		\
	__PMC_EV(PPC7450, LSU_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, LOAD_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, SS_SM_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, TLBIE_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, LWARX_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, MFSPR_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, REFETCH_SERIALIZATION)			\
	__PMC_EV(PPC7450, COMPLETION_QUEUE_ENTRIES_OVER_THRESHOLD)	\
	__PMC_EV(PPC7450, CYCLES_ONE_INSTR_DISPATCHED)			\
	__PMC_EV(PPC7450, CYCLES_TWO_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, ITLB_NON_SPECULATIVE_MISSES)			\
	__PMC_EV(PPC7450, CYCLES_WAITING_FROM_L1_INSTR_CACHE_MISS)	\
	__PMC_EV(PPC7450, L1_DATA_LOAD_ACCESS_MISS)			\
	__PMC_EV(PPC7450, L1_DATA_TOUCH_MISS)				\
	__PMC_EV(PPC7450, L1_DATA_STORE_MISS)				\
	__PMC_EV(PPC7450, L1_DATA_TOUCH_MISS_CYCLES)			\
	__PMC_EV(PPC7450, L1_DATA_CYCLES_USED)				\
	__PMC_EV(PPC7450, DST_STREAM_1_CACHE_LINE_FETCHES)		\
	__PMC_EV(PPC7450, VTQ_STREAM_CANCELED_PREMATURELY)		\
	__PMC_EV(PPC7450, VTQ_RESUMES_DUE_TO_CTX_CHANGE)		\
	__PMC_EV(PPC7450, VTQ_LINE_FETCH_MISS)				\
	__PMC_EV(PPC7450, VTQ_LINE_FETCH)				\
	__PMC_EV(PPC7450, TLBIE_SNOOPS)					\
	__PMC_EV(PPC7450, L1_INSTR_CACHE_RELOADS)			\
	__PMC_EV(PPC7450, L1_DATA_CACHE_RELOADS)			\
	__PMC_EV(PPC7450, L1_DATA_CACHE_CASTOUTS_TO_L2)			\
	__PMC_EV(PPC7450, STORE_MERGE_GATHER)				\
	__PMC_EV(PPC7450, CACHEABLE_STORE_MERGE_TO_32_BYTES)		\
	__PMC_EV(PPC7450, DATA_BKPT_MATCHES)				\
	__PMC_EV(PPC7450, FALL_THROUGH_BRANCHES_PROCESSED)		\
	__PMC_EV(PPC7450,						\
	    FIRST_SPECULATIVE_BRANCH_BUFFER_RESOLVED_CORRECTLY)		\
	__PMC_EV(PPC7450, SECOND_SPECULATION_BUFFER_ACTIVE)		\
	__PMC_EV(PPC7450, BPU_STALL_ON_LR_DEPENDENCY)			\
	__PMC_EV(PPC7450, BTIC_MISS)					\
	__PMC_EV(PPC7450, BRANCH_LINK_STACK_CORRECTLY_RESOLVED)		\
	__PMC_EV(PPC7450, FPR_ISSUE_STALLED)				\
	__PMC_EV(PPC7450, SWITCHES_BETWEEN_PRIV_USER)			\
	__PMC_EV(PPC7450, LSU_COMPLETES_FP_STORE_SINGLE)		\
	__PMC_EV(PPC7450, VR_ISSUE_QUEUE_DISPATCHES)			\
	__PMC_EV(PPC7450, VR_STALLS)					\
	__PMC_EV(PPC7450, GPR_RENAME_BUFFER_ENTRIES_OVER_THRESHOLD)	\
	__PMC_EV(PPC7450, FPR_ISSUE_QUEUE_ENTRIES)			\
	__PMC_EV(PPC7450, FPU_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, STWCX_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, LS_LM_INSTR_PIECES)				\
	__PMC_EV(PPC7450, ITLB_HW_SEARCH_CYCLES_OVER_THRESHOLD)		\
	__PMC_EV(PPC7450, DTLB_MISSES)					\
	__PMC_EV(PPC7450, CANCELLED_L1_INSTR_CACHE_MISSES)		\
	__PMC_EV(PPC7450, L1_DATA_CACHE_OP_HIT)				\
	__PMC_EV(PPC7450, L1_DATA_LOAD_MISS_CYCLES)			\
	__PMC_EV(PPC7450, L1_DATA_PUSHES)				\
	__PMC_EV(PPC7450, L1_DATA_TOTAL_MISS)				\
	__PMC_EV(PPC7450, VT2_FETCHES)					\
	__PMC_EV(PPC7450, TAKEN_BRANCHES_PROCESSED)			\
	__PMC_EV(PPC7450, BRANCH_FLUSHES)				\
	__PMC_EV(PPC7450,						\
	    SECOND_SPECULATIVE_BRANCH_BUFFER_RESOLVED_CORRECTLY)	\
	__PMC_EV(PPC7450, THIRD_SPECULATION_BUFFER_ACTIVE)		\
	__PMC_EV(PPC7450, BRANCH_UNIT_STALL_ON_CTR_DEPENDENCY)		\
	__PMC_EV(PPC7450, FAST_BTIC_HIT)				\
	__PMC_EV(PPC7450, BRANCH_LINK_STACK_MISPREDICTED)		\
	__PMC_EV(PPC7450, CYCLES_THREE_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, CYCLES_NO_INSTR_DISPATCHED)			\
	__PMC_EV(PPC7450, GPR_ISSUE_QUEUE_ENTRIES_OVER_THRESHOLD)	\
	__PMC_EV(PPC7450, GPR_ISSUE_QUEUE_STALLED)			\
	__PMC_EV(PPC7450, IU1_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, DSSALL_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, TLBSYNC_INSTR_COMPLETED)			\
	__PMC_EV(PPC7450, SYNC_INSTR_COMPLETED)				\
	__PMC_EV(PPC7450, SS_SM_INSTR_PIECES)				\
	__PMC_EV(PPC7450, DTLB_HW_SEARCH_CYCLES)			\
	__PMC_EV(PPC7450, SNOOP_RETRIES)				\
	__PMC_EV(PPC7450, SUCCESSFUL_STWCX)				\
	__PMC_EV(PPC7450, DST_STREAM_3_CACHE_LINE_FETCHES)		\
	__PMC_EV(PPC7450,						\
	    THIRD_SPECULATIVE_BRANCH_BUFFER_RESOLVED_CORRECTLY)		\
	__PMC_EV(PPC7450, MISPREDICTED_BRANCHES)			\
	__PMC_EV(PPC7450, FOLDED_BRANCHES)				\
	__PMC_EV(PPC7450, FP_STORE_DOUBLE_COMPLETES_IN_LSU)		\
	__PMC_EV(PPC7450, L2_CACHE_HITS)				\
	__PMC_EV(PPC7450, L3_CACHE_HITS)				\
	__PMC_EV(PPC7450, L2_INSTR_CACHE_MISSES)			\
	__PMC_EV(PPC7450, L3_INSTR_CACHE_MISSES)			\
	__PMC_EV(PPC7450, L2_DATA_CACHE_MISSES)				\
	__PMC_EV(PPC7450, L3_DATA_CACHE_MISSES)				\
	__PMC_EV(PPC7450, L2_LOAD_HITS)					\
	__PMC_EV(PPC7450, L2_STORE_HITS)				\
	__PMC_EV(PPC7450, L3_LOAD_HITS)					\
	__PMC_EV(PPC7450, L3_STORE_HITS)				\
	__PMC_EV(PPC7450, L2_TOUCH_HITS)				\
	__PMC_EV(PPC7450, L3_TOUCH_HITS)				\
	__PMC_EV(PPC7450, SNOOP_MODIFIED)				\
	__PMC_EV(PPC7450, SNOOP_VALID)					\
	__PMC_EV(PPC7450, INTERVENTION)					\
	__PMC_EV(PPC7450, L2_CACHE_MISSES)				\
	__PMC_EV(PPC7450, L3_CACHE_MISSES)				\
	__PMC_EV(PPC7450, L2_CACHE_CASTOUTS)				\
	__PMC_EV(PPC7450, L3_CACHE_CASTOUTS)				\
	__PMC_EV(PPC7450, L2SQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, L3SQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, RAQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, WAQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, L1_EXTERNAL_INTERVENTIONS)			\
	__PMC_EV(PPC7450, L2_EXTERNAL_INTERVENTIONS)			\
	__PMC_EV(PPC7450, L3_EXTERNAL_INTERVENTIONS)			\
	__PMC_EV(PPC7450, EXTERNAL_INTERVENTIONS)			\
	__PMC_EV(PPC7450, EXTERNAL_PUSHES)				\
	__PMC_EV(PPC7450, EXTERNAL_SNOOP_RETRY)				\
	__PMC_EV(PPC7450, DTQ_FULL_CYCLES)				\
	__PMC_EV(PPC7450, BUS_RETRY)					\
	__PMC_EV(PPC7450, L2_VALID_REQUEST)				\
	__PMC_EV(PPC7450, BORDQ_FULL)					\
	__PMC_EV(PPC7450, BUS_TAS_FOR_READS)				\
	__PMC_EV(PPC7450, BUS_TAS_FOR_WRITES)				\
	__PMC_EV(PPC7450, BUS_READS_NOT_RETRIED)			\
	__PMC_EV(PPC7450, BUS_WRITES_NOT_RETRIED)			\
	__PMC_EV(PPC7450, BUS_READS_WRITES_NOT_RETRIED)			\
	__PMC_EV(PPC7450, BUS_RETRY_DUE_TO_L1_RETRY)			\
	__PMC_EV(PPC7450, BUS_RETRY_DUE_TO_PREVIOUS_ADJACENT)		\
	__PMC_EV(PPC7450, BUS_RETRY_DUE_TO_COLLISION)			\
	__PMC_EV(PPC7450, BUS_RETRY_DUE_TO_INTERVENTION_ORDERING)	\
	__PMC_EV(PPC7450, SNOOP_REQUESTS)				\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_REQUEST)			\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_COLLISION_VS_LOAD)		\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_COLLISION_VS_STORE)		\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_COLLISION_VS_INSTR_FETCH)	\
	__PMC_EV(PPC7450,						\
	    PREFETCH_ENGINE_COLLISION_VS_LOAD_STORE_INSTR_FETCH)	\
	__PMC_EV(PPC7450, PREFETCH_ENGINE_FULL)

#define PMC_EV_PPC7450_FIRST	PMC_EV_PPC7450_CYCLE
#define PMC_EV_PPC7450_LAST	PMC_EV_PPC7450_PREFETCH_ENGINE_FULL

#define __PMC_EV_PPC970() \
	__PMC_EV(PPC970, INSTR_COMPLETED) \
	__PMC_EV(PPC970, MARKED_GROUP_DISPATCH) \
	__PMC_EV(PPC970, MARKED_STORE_COMPLETED) \
	__PMC_EV(PPC970, GCT_EMPTY) \
	__PMC_EV(PPC970, RUN_CYCLES) \
	__PMC_EV(PPC970, OVERFLOW) \
	__PMC_EV(PPC970, CYCLES) \
	__PMC_EV(PPC970, THRESHOLD_TIMEOUT) \
	__PMC_EV(PPC970, GROUP_DISPATCH) \
	__PMC_EV(PPC970, BR_MARKED_INSTR_FINISH) \
	__PMC_EV(PPC970, GCT_EMPTY_BY_SRQ_FULL) \
	__PMC_EV(PPC970, STOP_COMPLETION) \
	__PMC_EV(PPC970, LSU_EMPTY) \
	__PMC_EV(PPC970, MARKED_STORE_WITH_INTR) \
	__PMC_EV(PPC970, CYCLES_IN_SUPER) \
	__PMC_EV(PPC970, VPU_MARKED_INSTR_COMPLETED) \
	__PMC_EV(PPC970, FXU0_IDLE_FXU1_BUSY) \
	__PMC_EV(PPC970, SRQ_EMPTY) \
	__PMC_EV(PPC970, MARKED_GROUP_COMPLETED) \
	__PMC_EV(PPC970, CR_MARKED_INSTR_FINISH) \
	__PMC_EV(PPC970, DISPATCH_SUCCESS) \
	__PMC_EV(PPC970, FXU0_IDLE_FXU1_IDLE) \
	__PMC_EV(PPC970, ONE_PLUS_INSTR_COMPLETED) \
	__PMC_EV(PPC970, GROUP_MARKED_IDU) \
	__PMC_EV(PPC970, MARKED_GROUP_COMPLETE_TIMEOUT) \
	__PMC_EV(PPC970, FXU0_BUSY_FXU1_BUSY) \
	__PMC_EV(PPC970, MARKED_STORE_SENT_TO_STS) \
	__PMC_EV(PPC970, FXU_MARKED_INSTR_FINISHED) \
	__PMC_EV(PPC970, MARKED_GROUP_ISSUED) \
	__PMC_EV(PPC970, FXU0_BUSY_FXU1_IDLE) \
	__PMC_EV(PPC970, GROUP_COMPLETED) \
	__PMC_EV(PPC970, FPU_MARKED_INSTR_COMPLETED) \
	__PMC_EV(PPC970, MARKED_INSTR_FINISH_ANY_UNIT) \
	__PMC_EV(PPC970, EXTERNAL_INTERRUPT) \
	__PMC_EV(PPC970, GROUP_DISPATCH_REJECT) \
	__PMC_EV(PPC970, LSU_MARKED_INSTR_FINISH) \
	__PMC_EV(PPC970, TIMEBASE_EVENT) \
	__PMC_EV(PPC970, LSU_COMPLETION_STALL) \
	__PMC_EV(PPC970, FXU_COMPLETION_STALL) \
	__PMC_EV(PPC970, DCACHE_MISS_COMPLETION_STALL) \
	__PMC_EV(PPC970, FPU_COMPLETION_STALL) \
	__PMC_EV(PPC970, FXU_LONG_INSTR_COMPLETION_STALL) \
	__PMC_EV(PPC970, REJECT_COMPLETION_STALL) \
	__PMC_EV(PPC970, FPU_LONG_INSTR_COMPLETION_STALL) \
	__PMC_EV(PPC970, GCT_EMPTY_BY_ICACHE_MISS) \
	__PMC_EV(PPC970, REJECT_COMPLETION_STALL_ERAT_MISS) \
	__PMC_EV(PPC970, GCT_EMPTY_BY_BRANCH_MISS_PREDICT) \
	__PMC_EV(PPC970, BUS_HIGH) \
	__PMC_EV(PPC970, BUS_LOW) \
	__PMC_EV(PPC970, ADDER)


#define PMC_EV_PPC970_FIRST		PMC_EV_PPC970_INSTR_COMPLETED
#define PMC_EV_PPC970_LAST		PMC_EV_PPC970_ADDER

#define __PMC_EV_E500() \
	__PMC_EV(E500, CYCLES) \
	__PMC_EV(E500, INSTR_COMPLETED) \
	__PMC_EV(E500, UOPS_COMPLETED) \
	__PMC_EV(E500, INSTR_FETCHED) \
	__PMC_EV(E500, UOPS_DECODED) \
	__PMC_EV(E500, PM_EVENT_TRANSITIONS) \
	__PMC_EV(E500, PM_EVENT_CYCLES) \
	__PMC_EV(E500, BRANCH_INSTRS_COMPLETED) \
	__PMC_EV(E500, LOAD_UOPS_COMPLETED) \
	__PMC_EV(E500, STORE_UOPS_COMPLETED) \
	__PMC_EV(E500, CQ_REDIRECTS) \
	__PMC_EV(E500, BRANCHES_FINISHED) \
	__PMC_EV(E500, TAKEN_BRANCHES_FINISHED) \
	__PMC_EV(E500, FINISHED_UNCOND_BRANCHES_MISS_BTB) \
	__PMC_EV(E500, BRANCH_MISPRED) \
	__PMC_EV(E500, BTB_BRANCH_MISPRED_FROM_DIRECTION) \
	__PMC_EV(E500, BTB_HITS_PSEUDO_HITS) \
	__PMC_EV(E500, CYCLES_DECODE_STALLED) \
	__PMC_EV(E500, CYCLES_ISSUE_STALLED) \
	__PMC_EV(E500, CYCLES_BRANCH_ISSUE_STALLED) \
	__PMC_EV(E500, CYCLES_SU1_SCHED_STALLED) \
	__PMC_EV(E500, CYCLES_SU2_SCHED_STALLED) \
	__PMC_EV(E500, CYCLES_MU_SCHED_STALLED) \
	__PMC_EV(E500, CYCLES_LRU_SCHED_STALLED) \
	__PMC_EV(E500, CYCLES_BU_SCHED_STALLED) \
	__PMC_EV(E500, TOTAL_TRANSLATED) \
	__PMC_EV(E500, LOADS_TRANSLATED) \
	__PMC_EV(E500, STORES_TRANSLATED) \
	__PMC_EV(E500, TOUCHES_TRANSLATED) \
	__PMC_EV(E500, CACHEOPS_TRANSLATED) \
	__PMC_EV(E500, CACHE_INHIBITED_ACCESS_TRANSLATED) \
	__PMC_EV(E500, GUARDED_LOADS_TRANSLATED) \
	__PMC_EV(E500, WRITE_THROUGH_STORES_TRANSLATED) \
	__PMC_EV(E500, MISALIGNED_LOAD_STORE_ACCESS_TRANSLATED) \
	__PMC_EV(E500, TOTAL_ALLOCATED_TO_DLFB) \
	__PMC_EV(E500, LOADS_TRANSLATED_ALLOCATED_TO_DLFB) \
	__PMC_EV(E500, STORES_COMPLETED_ALLOCATED_TO_DLFB) \
	__PMC_EV(E500, TOUCHES_TRANSLATED_ALLOCATED_TO_DLFB) \
	__PMC_EV(E500, STORES_COMPLETED) \
	__PMC_EV(E500, DATA_L1_CACHE_LOCKS) \
	__PMC_EV(E500, DATA_L1_CACHE_RELOADS) \
	__PMC_EV(E500, DATA_L1_CACHE_CASTOUTS) \
	__PMC_EV(E500, LOAD_MISS_DLFB_FULL) \
	__PMC_EV(E500, LOAD_MISS_LDQ_FULL) \
	__PMC_EV(E500, LOAD_GUARDED_MISS) \
	__PMC_EV(E500, STORE_TRANSLATE_WHEN_QUEUE_FULL) \
	__PMC_EV(E500, ADDRESS_COLLISION) \
	__PMC_EV(E500, DATA_MMU_MISS) \
	__PMC_EV(E500, DATA_MMU_BUSY) \
	__PMC_EV(E500, PART2_MISALIGNED_CACHE_ACCESS) \
	__PMC_EV(E500, LOAD_MISS_DLFB_FULL_CYCLES) \
	__PMC_EV(E500, LOAD_MISS_LDQ_FULL_CYCLES) \
	__PMC_EV(E500, LOAD_GUARDED_MISS_CYCLES) \
	__PMC_EV(E500, STORE_TRANSLATE_WHEN_QUEUE_FULL_CYCLES) \
	__PMC_EV(E500, ADDRESS_COLLISION_CYCLES) \
	__PMC_EV(E500, DATA_MMU_MISS_CYCLES) \
	__PMC_EV(E500, DATA_MMU_BUSY_CYCLES) \
	__PMC_EV(E500, PART2_MISALIGNED_CACHE_ACCESS_CYCLES) \
	__PMC_EV(E500, INSTR_L1_CACHE_LOCKS) \
	__PMC_EV(E500, INSTR_L1_CACHE_RELOADS) \
	__PMC_EV(E500, INSTR_L1_CACHE_FETCHES) \
	__PMC_EV(E500, INSTR_MMU_TLB4K_RELOADS) \
	__PMC_EV(E500, INSTR_MMU_VSP_RELOADS) \
	__PMC_EV(E500, DATA_MMU_TLB4K_RELOADS) \
	__PMC_EV(E500, DATA_MMU_VSP_RELOADS) \
	__PMC_EV(E500, L2MMU_MISSES) \
	__PMC_EV(E500, BIU_MASTER_REQUESTS) \
	__PMC_EV(E500, BIU_MASTER_INSTR_SIDE_REQUESTS) \
	__PMC_EV(E500, BIU_MASTER_DATA_SIDE_REQUESTS) \
	__PMC_EV(E500, BIU_MASTER_DATA_SIDE_CASTOUT_REQUESTS) \
	__PMC_EV(E500, BIU_MASTER_RETRIES) \
	__PMC_EV(E500, SNOOP_REQUESTS) \
	__PMC_EV(E500, SNOOP_HITS) \
	__PMC_EV(E500, SNOOP_PUSHES) \
	__PMC_EV(E500, SNOOP_RETRIES) \
	__PMC_EV(E500, DLFB_LOAD_MISS_CYCLES) \
	__PMC_EV(E500, ILFB_FETCH_MISS_CYCLES) \
	__PMC_EV(E500, EXT_INPU_INTR_LATENCY_CYCLES) \
	__PMC_EV(E500, CRIT_INPUT_INTR_LATENCY_CYCLES) \
	__PMC_EV(E500, EXT_INPUT_INTR_PENDING_LATENCY_CYCLES) \
	__PMC_EV(E500, CRIT_INPUT_INTR_PENDING_LATENCY_CYCLES) \
	__PMC_EV(E500, PMC0_OVERFLOW) \
	__PMC_EV(E500, PMC1_OVERFLOW) \
	__PMC_EV(E500, PMC2_OVERFLOW) \
	__PMC_EV(E500, PMC3_OVERFLOW) \
	__PMC_EV(E500, INTERRUPTS_TAKEN) \
	__PMC_EV(E500, EXT_INPUT_INTR_TAKEN) \
	__PMC_EV(E500, CRIT_INPUT_INTR_TAKEN) \
	__PMC_EV(E500, SYSCALL_TRAP_INTR) \
	__PMC_EV(E500, TLB_BIT_TRANSITIONS) \
	__PMC_EV(E500, L2_LINEFILL_BUFFER) \
	__PMC_EV(E500, LV2_VS) \
	__PMC_EV(E500, CASTOUTS_RELEASED) \
	__PMC_EV(E500, INTV_ALLOCATIONS) \
	__PMC_EV(E500, DLFB_RETRIES_TO_MBAR) \
	__PMC_EV(E500, STORE_RETRIES) \
	__PMC_EV(E500, STASH_L1_HITS) \
	__PMC_EV(E500, STASH_L2_HITS) \
	__PMC_EV(E500, STASH_BUSY_1) \
	__PMC_EV(E500, STASH_BUSY_2) \
	__PMC_EV(E500, STASH_BUSY_3) \
	__PMC_EV(E500, STASH_HITS) \
	__PMC_EV(E500, STASH_HIT_DLFB) \
	__PMC_EV(E500, STASH_REQUESTS) \
	__PMC_EV(E500, STASH_REQUESTS_L1) \
	__PMC_EV(E500, STASH_REQUESTS_L2) \
	__PMC_EV(E500, STALLS_NO_CAQ_OR_COB) \
	__PMC_EV(E500, L2_CACHE_ACCESSES) \
	__PMC_EV(E500, L2_HIT_CACHE_ACCESSES) \
	__PMC_EV(E500, L2_CACHE_DATA_ACCESSES) \
	__PMC_EV(E500, L2_CACHE_DATA_HITS) \
	__PMC_EV(E500, L2_CACHE_INSTR_ACCESSES) \
	__PMC_EV(E500, L2_CACHE_INSTR_HITS) \
	__PMC_EV(E500, L2_CACHE_ALLOCATIONS) \
	__PMC_EV(E500, L2_CACHE_DATA_ALLOCATIONS) \
	__PMC_EV(E500, L2_CACHE_DIRTY_DATA_ALLOCATIONS) \
	__PMC_EV(E500, L2_CACHE_INSTR_ALLOCATIONS) \
	__PMC_EV(E500, L2_CACHE_UPDATES) \
	__PMC_EV(E500, L2_CACHE_CLEAN_UPDATES) \
	__PMC_EV(E500, L2_CACHE_DIRTY_UPDATES) \
	__PMC_EV(E500, L2_CACHE_CLEAN_REDUNDANT_UPDATES) \
	__PMC_EV(E500, L2_CACHE_DIRTY_REDUNDANT_UPDATES) \
	__PMC_EV(E500, L2_CACHE_LOCKS) \
	__PMC_EV(E500, L2_CACHE_CASTOUTS) \
	__PMC_EV(E500, L2_CACHE_DATA_DIRTY_HITS) \
	__PMC_EV(E500, INSTR_LFB_WENT_HIGH_PRIORITY) \
	__PMC_EV(E500, SNOOP_THROTTLING_TURNED_ON) \
	__PMC_EV(E500, L2_CLEAN_LINE_INVALIDATIONS) \
	__PMC_EV(E500, L2_INCOHERENT_LINE_INVALIDATIONS) \
	__PMC_EV(E500, L2_COHERENT_LINE_INVALIDATIONS) \
	__PMC_EV(E500, COHERENT_LOOKUP_MISS_DUE_TO_VALID_BUT_INCOHERENT_MATCHES) \
	__PMC_EV(E500, IAC1S_DETECTED) \
	__PMC_EV(E500, IAC2S_DETECTED) \
	__PMC_EV(E500, DAC1S_DTECTED) \
	__PMC_EV(E500, DAC2S_DTECTED) \
	__PMC_EV(E500, DVT0_DETECTED) \
	__PMC_EV(E500, DVT1_DETECTED) \
	__PMC_EV(E500, DVT2_DETECTED) \
	__PMC_EV(E500, DVT3_DETECTED) \
	__PMC_EV(E500, DVT4_DETECTED) \
	__PMC_EV(E500, DVT5_DETECTED) \
	__PMC_EV(E500, DVT6_DETECTED) \
	__PMC_EV(E500, DVT7_DETECTED) \
	__PMC_EV(E500, CYCLES_COMPLETION_STALLED_NEXUS_FIFO_FULL) \
	__PMC_EV(E500, FPU_DOUBLE_PUMP) \
	__PMC_EV(E500, FPU_FINISH) \
	__PMC_EV(E500, FPU_DIVIDE_CYCLES) \
	__PMC_EV(E500, FPU_DENORM_INPUT_CYCLES) \
	__PMC_EV(E500, FPU_RESULT_STALL_CYCLES) \
	__PMC_EV(E500, FPU_FPSCR_FULL_STALL) \
	__PMC_EV(E500, FPU_PIPE_SYNC_STALLS) \
	__PMC_EV(E500, FPU_INPUT_DATA_STALLS) \
	__PMC_EV(E500, DECORATED_LOADS) \
	__PMC_EV(E500, DECORATED_STORES) \
	__PMC_EV(E500, LOAD_RETRIES) \
	__PMC_EV(E500, STWCX_SUCCESSES) \
	__PMC_EV(E500, STWCX_FAILURES) \

#define PMC_EV_E500_FIRST		PMC_EV_E500_CYCLES
#define PMC_EV_E500_LAST		PMC_EV_E500_STWCX_FAILURES
/*
 * All known PMC events.
 *
 * PMC event numbers are allocated sparsely to allow new PMC events to
 * be added to a PMC class without breaking ABI compatibility.  The
 * current allocation scheme is:
 *
 * START	#EVENTS		DESCRIPTION
 * 0		0x1000		Reserved
 * 0x1000	0x0001		TSC
 * 0x2000	0x0080		free (was AMD K7 events)
 * 0x2080	0x0100		AMD K8 events
 * 0x10000	0x0080		INTEL architectural fixed-function events
 * 0x10080	0x0F80		free (was INTEL architectural programmable events)
 * 0x11000	0x0080		free (was INTEL Pentium 4 events)
 * 0x11080	0x0080		free (was INTEL Pentium MMX events)
 * 0x11100	0x0100		free (was INTEL Pentium Pro/P-II/P-III/Pentium-M events)
 * 0x11200	0x00FF		free (was INTEL XScale events)
 * 0x11300	0x00FF		free (was MIPS 24K events)
 * 0x11400	0x00FF		free (was Octeon events)
 * 0x11500	0x00FF		free (was MIPS 74K events)
 * 0x11600	0x00FF		free (was BERI statcounters)
 * 0x13000	0x00FF		MPC7450 events
 * 0x13100	0x00FF		IBM PPC970 events
 * 0x13200	0x00FF		free (was IBM POWER8 events)
 * 0x13300	0x00FF		Freescale e500 events
 * 0x14000	0x0100		ARMv7 events
 * 0x14100	0x0400		ARMv8 events
 * 0x14500	0x0020		ARM DMC-620 clkdiv2 events
 * 0x14520	0x0080		ARM DMC-620 clk events
 * 0x14600	0x0100		ARM CMN-600 events
 * 0x20000	0x1000		Software events
 */
#define	__PMC_EVENTS()					\
	__PMC_EV_BLOCK(TSC,		0x01000)	\
	__PMC_EV_TSC()					\
	__PMC_EV_BLOCK(K8,		0x02080)	\
	__PMC_EV_K8()					\
	__PMC_EV_BLOCK(IAF,		0x10000)	\
	__PMC_EV_IAF()					\
	__PMC_EV_BLOCK(PPC7450,		0x13000)	\
	__PMC_EV_PPC7450()				\
	__PMC_EV_BLOCK(PPC970,		0x13100)	\
	__PMC_EV_PPC970()				\
	__PMC_EV_BLOCK(E500,		0x13300)	\
	__PMC_EV_E500()					\
	__PMC_EV_BLOCK(ARMV7,		0x14000)	\
	__PMC_EV_ARMV7()				\
	__PMC_EV_BLOCK(ARMV8,		0x14100)	\
	__PMC_EV_ARMV8()				\
	__PMC_EV_BLOCK(DMC620_PMU_CD2,	0x14500)	\
	__PMC_EV_DMC620_PMU_CD2()			\
	__PMC_EV_BLOCK(DMC620_PMU_C,	0x14520)	\
	__PMC_EV_DMC620_PMU_C()				\
	__PMC_EV_BLOCK(CMN600_PMU,	0x14600)	\
	__PMC_EV_CMN600_PMU()

#define	PMC_EVENT_FIRST	PMC_EV_TSC_TSC
#define	PMC_EVENT_LAST	PMC_EV_SOFT_LAST

#endif /* _DEV_HWPMC_PMC_EVENTS_H_ */
