package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link Preference}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutablePreference.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutablePreference implements Preference {
  private final Spread spread;

  private ImmutablePreference(Spread spread) {
    this.spread = spread;
  }

  /**
   * @return The value of the {@code spread} attribute
   */
  @JsonProperty("Spread")
  @Override
  public Spread spread() {
    return spread;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Preference#spread() spread} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for spread
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePreference withSpread(Spread value) {
    if (this.spread == value) return this;
    Spread newValue = Objects.requireNonNull(value, "spread");
    return new ImmutablePreference(newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePreference} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePreference
        && equalsByValue((ImmutablePreference) another);
  }

  private boolean equalsByValue(ImmutablePreference another) {
    return spread.equals(another.spread);
  }

  /**
   * Computes a hash code from attributes: {@code spread}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + spread.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Preference} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Preference{"
        + "spread=" + spread
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Preference} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Preference instance
   */
  public static ImmutablePreference copyOf(Preference instance) {
    if (instance instanceof ImmutablePreference) {
      return (ImmutablePreference) instance;
    }
    return ImmutablePreference.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutablePreference ImmutablePreference}.
   * <pre>
   * ImmutablePreference.builder()
   *    .spread(org.mandas.docker.client.messages.swarm.Spread) // required {@link Preference#spread() spread}
   *    .build();
   * </pre>
   * @return A new ImmutablePreference builder
   */
  public static ImmutablePreference.Builder builder() {
    return new ImmutablePreference.Builder();
  }

  /**
   * Builds instances of type {@link ImmutablePreference ImmutablePreference}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_SPREAD = 0x1L;
    private long initBits = 0x1L;

    private Spread spread;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Preference} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Preference instance) {
      Objects.requireNonNull(instance, "instance");
      this.spread(instance.spread());
      return this;
    }

    /**
     * Initializes the value for the {@link Preference#spread() spread} attribute.
     * @param spread The value for spread 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Spread")
    public final Builder spread(Spread spread) {
      this.spread = Objects.requireNonNull(spread, "spread");
      initBits &= ~INIT_BIT_SPREAD;
      return this;
    }

    /**
     * Builds a new {@link ImmutablePreference ImmutablePreference}.
     * @return An immutable instance of Preference
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutablePreference build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutablePreference(spread);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_SPREAD) != 0) attributes.add("spread");
      return "Cannot build Preference, some of required attributes are not set " + attributes;
    }
  }
}
